package com.caucho.hessian.io;

import com.caucho.hessian.io.AbstractHessianInput;
import com.caucho.hessian.io.Deserializer;
import com.caucho.hessian.io.Hessian2Constants;
import com.caucho.hessian.io.HessianDebugInputStream;
import com.caucho.hessian.io.HessianProtocolException;
import com.caucho.hessian.io.HessianRemote;
import com.caucho.hessian.io.HessianRemoteResolver;
import com.caucho.hessian.io.HessianServiceException;
import com.caucho.hessian.io.SerializerFactory;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.w3c.dom.Node;

public class Hessian2Input extends AbstractHessianInput implements Hessian2Constants {

   private static final Logger log = Logger.getLogger(Hessian2Input.class.getName());
   private static final int END_OF_DATA = -2;
   private static Field _detailMessageField;
   private static final int SIZE = 1024;
   private static final int GAP = 16;
   protected SerializerFactory _defaultSerializerFactory;
   protected SerializerFactory _serializerFactory;
   private static boolean _isCloseStreamOnClose;
   protected ArrayList _refs;
   protected ArrayList _classDefs;
   protected ArrayList _types;
   private InputStream _is;
   private final byte[] _buffer;
   private int _offset;
   private int _length;
   private String _method;
   private Throwable _replyFault;
   private StringBuilder _sbuf;
   private boolean _isLastChunk;
   private int _chunkLength;
   private HessianDebugInputStream _dIs;


   public Hessian2Input() {
      this._refs = new ArrayList();
      this._classDefs = new ArrayList();
      this._types = new ArrayList();
      this._buffer = new byte[1024];
      this._sbuf = new StringBuilder();
      if(log.isLoggable(Level.FINEST)) {
         this._dIs = new HessianDebugInputStream(log, Level.FINEST);
      }

   }

   public Hessian2Input(InputStream is) {
      this();
      this.init(is);
   }

   public void setSerializerFactory(SerializerFactory factory) {
      this._serializerFactory = factory;
   }

   public SerializerFactory getSerializerFactory() {
      if(this._serializerFactory == this._defaultSerializerFactory) {
         this._serializerFactory = new SerializerFactory();
      }

      return this._serializerFactory;
   }

   protected final SerializerFactory findSerializerFactory() {
      SerializerFactory factory = this._serializerFactory;
      if(factory == null) {
         factory = SerializerFactory.createDefault();
         this._defaultSerializerFactory = factory;
         this._serializerFactory = factory;
      }

      return factory;
   }

   public void setCloseStreamOnClose(boolean isClose) {
      _isCloseStreamOnClose = isClose;
   }

   public boolean isCloseStreamOnClose() {
      return _isCloseStreamOnClose;
   }

   public String getMethod() {
      return this._method;
   }

   public Throwable getReplyFault() {
      return this._replyFault;
   }

   public void init(InputStream is) {
      if(this._dIs != null) {
         this._dIs.initPacket((InputStream)is);
         is = this._dIs;
      }

      this._is = (InputStream)is;
      this.reset();
   }

   public void initPacket(InputStream is) {
      if(this._dIs != null) {
         this._dIs.initPacket((InputStream)is);
         is = this._dIs;
      }

      this._is = (InputStream)is;
      this.resetReferences();
   }

   public int readCall() throws IOException {
      int tag = this.read();
      if(tag != 67) {
         throw this.error("expected hessian call (\'C\') at " + this.codeName(tag));
      } else {
         return 0;
      }
   }

   public int readEnvelope() throws IOException {
      int tag = this.read();
      int version = 0;
      if(tag == 72) {
         int major = this.read();
         int minor = this.read();
         version = (major << 16) + minor;
         tag = this.read();
      }

      if(tag != 69) {
         throw this.error("expected hessian Envelope (\'E\') at " + this.codeName(tag));
      } else {
         return version;
      }
   }

   public void completeEnvelope() throws IOException {
      int tag = this.read();
      if(tag != 90) {
         this.error("expected end of envelope at " + this.codeName(tag));
      }

   }

   public String readMethod() throws IOException {
      this._method = this.readString();
      return this._method;
   }

   public int readMethodArgLength() throws IOException {
      return this.readInt();
   }

   public void startCall() throws IOException {
      this.readCall();
      this.readMethod();
   }

   public Object[] readArguments() throws IOException {
      int len = this.readInt();
      Object[] args = new Object[len];

      for(int i = 0; i < len; ++i) {
         args[i] = this.readObject();
      }

      return args;
   }

   public void completeCall() throws IOException {
   }

   public Object readReply(Class expectedClass) throws Throwable {
      int tag = this.read();
      if(tag == 82) {
         return this.readObject(expectedClass);
      } else if(tag == 70) {
         HashMap sb1 = (HashMap)this.readObject(HashMap.class);
         throw this.prepareFault(sb1);
      } else {
         StringBuilder sb = new StringBuilder();
         sb.append((char)tag);

         int e;
         try {
            while((e = this.read()) >= 0) {
               sb.append((char)e);
            }
         } catch (IOException var5) {
            log.log(Level.FINE, var5.toString(), var5);
         }

         throw this.error("expected hessian reply at " + this.codeName(tag) + "\n" + sb);
      }
   }

   public void startReply() throws Throwable {
      this.readReply(Object.class);
   }

   private Throwable prepareFault(HashMap fault) throws IOException {
      Object detail = fault.get("detail");
      String message = (String)fault.get("message");
      if(detail instanceof Throwable) {
         this._replyFault = (Throwable)detail;
         if(message != null && _detailMessageField != null) {
            try {
               _detailMessageField.set(this._replyFault, message);
            } catch (Throwable var5) {
               ;
            }
         }

         return this._replyFault;
      } else {
         String code = (String)fault.get("code");
         this._replyFault = new HessianServiceException(message, code, detail);
         return this._replyFault;
      }
   }

   public void completeReply() throws IOException {
   }

   public void completeValueReply() throws IOException {
      int tag = this.read();
      if(tag != 90) {
         this.error("expected end of reply at " + this.codeName(tag));
      }

   }

   public String readHeader() throws IOException {
      return null;
   }

   public int startMessage() throws IOException {
      int tag = this.read();
      if(tag != 112 && tag != 80) {
         throw this.error("expected Hessian message (\'p\') at " + this.codeName(tag));
      } else {
         int major = this.read();
         int minor = this.read();
         return (major << 16) + minor;
      }
   }

   public void completeMessage() throws IOException {
      int tag = this.read();
      if(tag != 90) {
         this.error("expected end of message at " + this.codeName(tag));
      }

   }

   public void readNull() throws IOException {
      int tag = this.read();
      switch(tag) {
      case 78:
         return;
      default:
         throw this.expect("null", tag);
      }
   }

   public boolean readBoolean() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int tag = var10000;
      switch(tag) {
      case 56:
      case 57:
      case 58:
      case 59:
      case 61:
      case 62:
      case 63:
         this.read();
         this.read();
         return true;
      case 60:
         return 256 * this.read() + this.read() != 0;
      case 64:
      case 65:
      case 66:
      case 67:
      case 69:
      case 71:
      case 72:
      case 74:
      case 75:
      case 77:
      case 79:
      case 80:
      case 81:
      case 82:
      case 83:
      case 85:
      case 86:
      case 87:
      case 88:
      case 90:
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
      default:
         throw this.expect("boolean", tag);
      case 68:
         return this.parseDouble() != 0.0D;
      case 70:
         return false;
      case 73:
         return this.parseInt() != 0;
      case 76:
         return this.parseLong() != 0L;
      case 78:
         return false;
      case 84:
         return true;
      case 89:
         return 16777216L * (long)this.read() + 65536L * (long)this.read() + (long)(256 * this.read()) + (long)this.read() != 0L;
      case 91:
         return false;
      case 92:
         return true;
      case 93:
         return this.read() != 0;
      case 94:
         return 256 * this.read() + this.read() != 0;
      case 95:
         int mills = this.parseInt();
         return mills != 0;
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return tag != 144;
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         this.read();
         return true;
      case 200:
         return this.read() != 0;
      case 208:
      case 209:
      case 210:
      case 211:
      case 213:
      case 214:
      case 215:
         this.read();
         this.read();
         return true;
      case 212:
         return 256 * this.read() + this.read() != 0;
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return tag != 224;
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         this.read();
         return true;
      case 248:
         return this.read() != 0;
      }
   }

   public short readShort() throws IOException {
      return (short)this.readInt();
   }

   public final int readInt() throws IOException {
      int tag = this.read();
      switch(tag) {
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
         return (tag - 60 << 16) + 256 * this.read() + this.read();
      case 64:
      case 65:
      case 66:
      case 67:
      case 69:
      case 71:
      case 72:
      case 74:
      case 75:
      case 77:
      case 79:
      case 80:
      case 81:
      case 82:
      case 83:
      case 85:
      case 86:
      case 87:
      case 88:
      case 90:
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
      default:
         throw this.expect("integer", tag);
      case 68:
         return (int)this.parseDouble();
      case 70:
         return 0;
      case 73:
      case 89:
         return (this.read() << 24) + (this.read() << 16) + (this.read() << 8) + this.read();
      case 76:
         return (int)this.parseLong();
      case 78:
         return 0;
      case 84:
         return 1;
      case 91:
         return 0;
      case 92:
         return 1;
      case 93:
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset];
         } else {
            var10000 = this.read();
         }

         return (byte)var10000;
      case 94:
         return (short)(256 * this.read() + this.read());
      case 95:
         int mills = this.parseInt();
         return (int)(0.001D * (double)mills);
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return tag - 144;
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 200:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         return (tag - 200 << 8) + this.read();
      case 208:
      case 209:
      case 210:
      case 211:
      case 212:
      case 213:
      case 214:
      case 215:
         return (tag - 212 << 16) + 256 * this.read() + this.read();
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return tag - 224;
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 248:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         return (tag - 248 << 8) + this.read();
      }
   }

   public long readLong() throws IOException {
      int tag = this.read();
      switch(tag) {
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
         return (long)((tag - 60 << 16) + 256 * this.read() + this.read());
      case 64:
      case 65:
      case 66:
      case 67:
      case 69:
      case 71:
      case 72:
      case 74:
      case 75:
      case 77:
      case 79:
      case 80:
      case 81:
      case 82:
      case 83:
      case 85:
      case 86:
      case 87:
      case 88:
      case 90:
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
      default:
         throw this.expect("long", tag);
      case 68:
         return (long)this.parseDouble();
      case 70:
         return 0L;
      case 73:
      case 89:
         return (long)this.parseInt();
      case 76:
         return this.parseLong();
      case 78:
         return 0L;
      case 84:
         return 1L;
      case 91:
         return 0L;
      case 92:
         return 1L;
      case 93:
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset];
         } else {
            var10000 = this.read();
         }

         return (long)((byte)var10000);
      case 94:
         return (long)((short)(256 * this.read() + this.read()));
      case 95:
         int mills = this.parseInt();
         return (long)(0.001D * (double)mills);
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return (long)(tag - 144);
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 200:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         return (long)((tag - 200 << 8) + this.read());
      case 208:
      case 209:
      case 210:
      case 211:
      case 212:
      case 213:
      case 214:
      case 215:
         return (long)((tag - 212 << 16) + 256 * this.read() + this.read());
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return (long)(tag - 224);
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 248:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         return (long)((tag - 248 << 8) + this.read());
      }
   }

   public float readFloat() throws IOException {
      return (float)this.readDouble();
   }

   public double readDouble() throws IOException {
      int tag = this.read();
      switch(tag) {
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
         return (double)((tag - 60 << 16) + 256 * this.read() + this.read());
      case 64:
      case 65:
      case 66:
      case 67:
      case 69:
      case 71:
      case 72:
      case 74:
      case 75:
      case 77:
      case 79:
      case 80:
      case 81:
      case 82:
      case 83:
      case 85:
      case 86:
      case 87:
      case 88:
      case 90:
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
      default:
         throw this.expect("double", tag);
      case 68:
         return this.parseDouble();
      case 70:
         return 0.0D;
      case 73:
      case 89:
         return (double)this.parseInt();
      case 76:
         return (double)this.parseLong();
      case 78:
         return 0.0D;
      case 84:
         return 1.0D;
      case 91:
         return 0.0D;
      case 92:
         return 1.0D;
      case 93:
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset];
         } else {
            var10000 = this.read();
         }

         return (double)((byte)var10000);
      case 94:
         return (double)((short)(256 * this.read() + this.read()));
      case 95:
         int mills = this.parseInt();
         return 0.001D * (double)mills;
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return (double)(tag - 144);
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 200:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         return (double)((tag - 200 << 8) + this.read());
      case 208:
      case 209:
      case 210:
      case 211:
      case 212:
      case 213:
      case 214:
      case 215:
         return (double)((tag - 212 << 16) + 256 * this.read() + this.read());
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return (double)(tag - 224);
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 248:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         return (double)((tag - 248 << 8) + this.read());
      }
   }

   public long readUTCDate() throws IOException {
      int tag = this.read();
      if(tag == 74) {
         return this.parseLong();
      } else if(tag == 75) {
         return (long)this.parseInt() * 60000L;
      } else {
         throw this.expect("date", tag);
      }
   }

   public int readChar() throws IOException {
      int tag;
      if(this._chunkLength > 0) {
         --this._chunkLength;
         if(this._chunkLength == 0 && this._isLastChunk) {
            this._chunkLength = -2;
         }

         tag = this.parseUTF8Char();
         return tag;
      } else if(this._chunkLength == -2) {
         this._chunkLength = 0;
         return -1;
      } else {
         tag = this.read();
         switch(tag) {
         case 78:
            return -1;
         case 82:
         case 83:
            this._isLastChunk = tag == 83;
            this._chunkLength = (this.read() << 8) + this.read();
            --this._chunkLength;
            int value = this.parseUTF8Char();
            if(this._chunkLength == 0 && this._isLastChunk) {
               this._chunkLength = -2;
            }

            return value;
         default:
            throw this.expect("char", tag);
         }
      }
   }

   public int readString(char[] buffer, int offset, int length) throws IOException {
      int readLength = 0;
      if(this._chunkLength == -2) {
         this._chunkLength = 0;
         return -1;
      } else {
         int tag;
         if(this._chunkLength == 0) {
            tag = this.read();
            switch(tag) {
            case 0:
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
            case 9:
            case 10:
            case 11:
            case 12:
            case 13:
            case 14:
            case 15:
            case 16:
            case 17:
            case 18:
            case 19:
            case 20:
            case 21:
            case 22:
            case 23:
            case 24:
            case 25:
            case 26:
            case 27:
            case 28:
            case 29:
            case 30:
            case 31:
               this._isLastChunk = true;
               this._chunkLength = tag - 0;
               break;
            case 32:
            case 33:
            case 34:
            case 35:
            case 36:
            case 37:
            case 38:
            case 39:
            case 40:
            case 41:
            case 42:
            case 43:
            case 44:
            case 45:
            case 46:
            case 47:
            case 52:
            case 53:
            case 54:
            case 55:
            case 56:
            case 57:
            case 58:
            case 59:
            case 60:
            case 61:
            case 62:
            case 63:
            case 64:
            case 65:
            case 66:
            case 67:
            case 68:
            case 69:
            case 70:
            case 71:
            case 72:
            case 73:
            case 74:
            case 75:
            case 76:
            case 77:
            case 79:
            case 80:
            case 81:
            default:
               throw this.expect("string", tag);
            case 48:
            case 49:
            case 50:
            case 51:
               this._isLastChunk = true;
               this._chunkLength = (tag - 48) * 256 + this.read();
               break;
            case 78:
               return -1;
            case 82:
            case 83:
               this._isLastChunk = tag == 83;
               this._chunkLength = (this.read() << 8) + this.read();
            }
         }

         while(length > 0) {
            if(this._chunkLength > 0) {
               buffer[offset++] = (char)this.parseUTF8Char();
               --this._chunkLength;
               --length;
               ++readLength;
            } else {
               if(this._isLastChunk) {
                  if(readLength == 0) {
                     return -1;
                  }

                  this._chunkLength = -2;
                  return readLength;
               }

               tag = this.read();
               switch(tag) {
               case 0:
               case 1:
               case 2:
               case 3:
               case 4:
               case 5:
               case 6:
               case 7:
               case 8:
               case 9:
               case 10:
               case 11:
               case 12:
               case 13:
               case 14:
               case 15:
               case 16:
               case 17:
               case 18:
               case 19:
               case 20:
               case 21:
               case 22:
               case 23:
               case 24:
               case 25:
               case 26:
               case 27:
               case 28:
               case 29:
               case 30:
               case 31:
                  this._isLastChunk = true;
                  this._chunkLength = tag - 0;
                  break;
               case 32:
               case 33:
               case 34:
               case 35:
               case 36:
               case 37:
               case 38:
               case 39:
               case 40:
               case 41:
               case 42:
               case 43:
               case 44:
               case 45:
               case 46:
               case 47:
               case 52:
               case 53:
               case 54:
               case 55:
               case 56:
               case 57:
               case 58:
               case 59:
               case 60:
               case 61:
               case 62:
               case 63:
               case 64:
               case 65:
               case 66:
               case 67:
               case 68:
               case 69:
               case 70:
               case 71:
               case 72:
               case 73:
               case 74:
               case 75:
               case 76:
               case 77:
               case 78:
               case 79:
               case 80:
               case 81:
               default:
                  throw this.expect("string", tag);
               case 48:
               case 49:
               case 50:
               case 51:
                  this._isLastChunk = true;
                  this._chunkLength = (tag - 48) * 256 + this.read();
                  break;
               case 82:
               case 83:
                  this._isLastChunk = tag == 83;
                  this._chunkLength = (this.read() << 8) + this.read();
               }
            }
         }

         if(readLength == 0) {
            return -1;
         } else if(this._chunkLength <= 0 && this._isLastChunk) {
            this._chunkLength = -2;
            return readLength;
         } else {
            return readLength;
         }
      }
   }

   public String readString() throws IOException {
      int tag = this.read();
      int ch;
      switch(tag) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      case 16:
      case 17:
      case 18:
      case 19:
      case 20:
      case 21:
      case 22:
      case 23:
      case 24:
      case 25:
      case 26:
      case 27:
      case 28:
      case 29:
      case 30:
      case 31:
         this._isLastChunk = true;
         this._chunkLength = tag - 0;
         this._sbuf.setLength(0);

         while((ch = this.parseChar()) >= 0) {
            this._sbuf.append((char)ch);
         }

         return this._sbuf.toString();
      case 32:
      case 33:
      case 34:
      case 35:
      case 36:
      case 37:
      case 38:
      case 39:
      case 40:
      case 41:
      case 42:
      case 43:
      case 44:
      case 45:
      case 46:
      case 47:
      case 52:
      case 53:
      case 54:
      case 55:
      case 64:
      case 65:
      case 66:
      case 67:
      case 69:
      case 71:
      case 72:
      case 74:
      case 75:
      case 77:
      case 79:
      case 80:
      case 81:
      case 85:
      case 86:
      case 87:
      case 88:
      case 90:
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
      default:
         throw this.expect("string", tag);
      case 48:
      case 49:
      case 50:
      case 51:
         this._isLastChunk = true;
         this._chunkLength = (tag - 48) * 256 + this.read();
         this._sbuf.setLength(0);

         while((ch = this.parseChar()) >= 0) {
            this._sbuf.append((char)ch);
         }

         return this._sbuf.toString();
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
         return String.valueOf((tag - 60 << 16) + 256 * this.read() + this.read());
      case 68:
         return String.valueOf(this.parseDouble());
      case 70:
         return "false";
      case 73:
      case 89:
         return String.valueOf(this.parseInt());
      case 76:
         return String.valueOf(this.parseLong());
      case 78:
         return null;
      case 82:
      case 83:
         this._isLastChunk = tag == 83;
         this._chunkLength = (this.read() << 8) + this.read();
         this._sbuf.setLength(0);

         while((ch = this.parseChar()) >= 0) {
            this._sbuf.append((char)ch);
         }

         return this._sbuf.toString();
      case 84:
         return "true";
      case 91:
         return "0.0";
      case 92:
         return "1.0";
      case 93:
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset];
         } else {
            var10000 = this.read();
         }

         return String.valueOf((byte)var10000);
      case 94:
         return String.valueOf((short)(256 * this.read() + this.read()));
      case 95:
         ch = this.parseInt();
         return String.valueOf(0.001D * (double)ch);
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return String.valueOf(tag - 144);
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 200:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         return String.valueOf((tag - 200 << 8) + this.read());
      case 208:
      case 209:
      case 210:
      case 211:
      case 212:
      case 213:
      case 214:
      case 215:
         return String.valueOf((tag - 212 << 16) + 256 * this.read() + this.read());
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return String.valueOf(tag - 224);
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 248:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         return String.valueOf((tag - 248 << 8) + this.read());
      }
   }

   public byte[] readBytes() throws IOException {
      int tag = this.read();
      byte[] buffer;
      int offset;
      int sublen;
      switch(tag) {
      case 32:
      case 33:
      case 34:
      case 35:
      case 36:
      case 37:
      case 38:
      case 39:
      case 40:
      case 41:
      case 42:
      case 43:
      case 44:
      case 45:
      case 46:
      case 47:
         this._isLastChunk = true;
         this._chunkLength = tag - 32;
         buffer = new byte[this._chunkLength];

         for(offset = 0; offset < this._chunkLength; offset += sublen) {
            sublen = this.read(buffer, 0, this._chunkLength - offset);
            if(sublen <= 0) {
               break;
            }
         }

         return buffer;
      case 48:
      case 49:
      case 50:
      case 51:
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
      case 64:
      case 67:
      case 68:
      case 69:
      case 70:
      case 71:
      case 72:
      case 73:
      case 74:
      case 75:
      case 76:
      case 77:
      default:
         throw this.expect("bytes", tag);
      case 52:
      case 53:
      case 54:
      case 55:
         this._isLastChunk = true;
         this._chunkLength = (tag - 52) * 256 + this.read();
         buffer = new byte[this._chunkLength];

         for(offset = 0; offset < this._chunkLength; offset += sublen) {
            sublen = this.read(buffer, 0, this._chunkLength - offset);
            if(sublen <= 0) {
               break;
            }
         }

         return buffer;
      case 65:
      case 66:
         this._isLastChunk = tag == 66;
         this._chunkLength = (this.read() << 8) + this.read();
         ByteArrayOutputStream bos = new ByteArrayOutputStream();

         int data;
         while((data = this.parseByte()) >= 0) {
            bos.write(data);
         }

         return bos.toByteArray();
      case 78:
         return null;
      }
   }

   public int readByte() throws IOException {
      if(this._chunkLength > 0) {
         --this._chunkLength;
         if(this._chunkLength == 0 && this._isLastChunk) {
            this._chunkLength = -2;
         }

         return this.read();
      } else if(this._chunkLength == -2) {
         this._chunkLength = 0;
         return -1;
      } else {
         int tag = this.read();
         int value;
         switch(tag) {
         case 32:
         case 33:
         case 34:
         case 35:
         case 36:
         case 37:
         case 38:
         case 39:
         case 40:
         case 41:
         case 42:
         case 43:
         case 44:
         case 45:
         case 46:
         case 47:
            this._isLastChunk = true;
            this._chunkLength = tag - 32;
            value = this.parseByte();
            if(this._chunkLength == 0) {
               this._chunkLength = -2;
            }

            return value;
         case 48:
         case 49:
         case 50:
         case 51:
         case 56:
         case 57:
         case 58:
         case 59:
         case 60:
         case 61:
         case 62:
         case 63:
         case 64:
         case 67:
         case 68:
         case 69:
         case 70:
         case 71:
         case 72:
         case 73:
         case 74:
         case 75:
         case 76:
         case 77:
         default:
            throw this.expect("binary", tag);
         case 52:
         case 53:
         case 54:
         case 55:
            this._isLastChunk = true;
            this._chunkLength = (tag - 52) * 256 + this.read();
            value = this.parseByte();
            if(this._chunkLength == 0) {
               this._chunkLength = -2;
            }

            return value;
         case 65:
         case 66:
            this._isLastChunk = tag == 66;
            this._chunkLength = (this.read() << 8) + this.read();
            value = this.parseByte();
            if(this._chunkLength == 0 && this._isLastChunk) {
               this._chunkLength = -2;
            }

            return value;
         case 78:
            return -1;
         }
      }
   }

   public int readBytes(byte[] buffer, int offset, int length) throws IOException {
      int readLength = 0;
      if(this._chunkLength == -2) {
         this._chunkLength = 0;
         return -1;
      } else {
         int tag;
         if(this._chunkLength == 0) {
            tag = this.read();
            switch(tag) {
            case 32:
            case 33:
            case 34:
            case 35:
            case 36:
            case 37:
            case 38:
            case 39:
            case 40:
            case 41:
            case 42:
            case 43:
            case 44:
            case 45:
            case 46:
            case 47:
               this._isLastChunk = true;
               this._chunkLength = tag - 32;
               break;
            case 48:
            case 49:
            case 50:
            case 51:
            case 56:
            case 57:
            case 58:
            case 59:
            case 60:
            case 61:
            case 62:
            case 63:
            case 64:
            case 67:
            case 68:
            case 69:
            case 70:
            case 71:
            case 72:
            case 73:
            case 74:
            case 75:
            case 76:
            case 77:
            default:
               throw this.expect("binary", tag);
            case 52:
            case 53:
            case 54:
            case 55:
               this._isLastChunk = true;
               this._chunkLength = (tag - 52) * 256 + this.read();
               break;
            case 65:
            case 66:
               this._isLastChunk = tag == 66;
               this._chunkLength = (this.read() << 8) + this.read();
               break;
            case 78:
               return -1;
            }
         }

         while(length > 0) {
            if(this._chunkLength > 0) {
               buffer[offset++] = (byte)this.read();
               --this._chunkLength;
               --length;
               ++readLength;
            } else {
               if(this._isLastChunk) {
                  if(readLength == 0) {
                     return -1;
                  }

                  this._chunkLength = -2;
                  return readLength;
               }

               tag = this.read();
               switch(tag) {
               case 65:
               case 66:
                  this._isLastChunk = tag == 66;
                  this._chunkLength = (this.read() << 8) + this.read();
                  break;
               default:
                  throw this.expect("binary", tag);
               }
            }
         }

         if(readLength == 0) {
            return -1;
         } else if(this._chunkLength <= 0 && this._isLastChunk) {
            this._chunkLength = -2;
            return readLength;
         } else {
            return readLength;
         }
      }
   }

   private HashMap readFault() throws IOException {
      HashMap map = new HashMap();

      int code;
      for(code = this.read(); code > 0 && code != 90; code = this.read()) {
         --this._offset;
         Object key = this.readObject();
         Object value = this.readObject();
         if(key != null && value != null) {
            map.put(key, value);
         }
      }

      if(code != 90) {
         throw this.expect("fault", code);
      } else {
         return map;
      }
   }

   public Object readObject(Class cl) throws IOException {
      if(cl != null && cl != Object.class) {
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset] & 255;
         } else {
            var10000 = this.read();
         }

         int tag = var10000;
         int value;
         Deserializer reader;
         Object v;
         Object v1;
         Deserializer var7;
         String var9;
         int var10;
         Deserializer var12;
         Hessian2Input.ObjectDefinition var13;
         switch(tag) {
         case 67:
            this.readObjectDefinition(cl);
            return this.readObject(cl);
         case 68:
         case 69:
         case 70:
         case 71:
         case 73:
         case 74:
         case 75:
         case 76:
         case 80:
         case 82:
         case 83:
         case 84:
         case 89:
         case 90:
         case 91:
         case 92:
         case 93:
         case 94:
         case 95:
         default:
            if(tag >= 0) {
               --this._offset;
            }

            Object var14 = this.findSerializerFactory().getDeserializer((Class)cl).readObject(this);
            return var14;
         case 72:
            var7 = this.findSerializerFactory().getDeserializer((Class)cl);
            return var7.readMap(this);
         case 77:
            var9 = this.readType();
            if("".equals(var9)) {
               reader = this.findSerializerFactory().getDeserializer((Class)cl);
               return reader.readMap(this);
            }

            reader = this.findSerializerFactory().getObjectDeserializer(var9, cl);
            return reader.readMap(this);
         case 78:
            return null;
         case 79:
            value = this.readInt();
            var10 = this._classDefs.size();
            if(value >= 0 && var10 > value) {
               var13 = (Hessian2Input.ObjectDefinition)this._classDefs.get(value);
               return this.readObjectInstance(cl, var13);
            }

            throw new HessianProtocolException("\'" + value + "\' is an unknown class definition");
         case 81:
            value = this.readInt();
            return this._refs.get(value);
         case 85:
            var9 = this.readType();
            reader = this.findSerializerFactory().getListDeserializer(var9, cl);
            v = reader.readList(this, -1);
            return v;
         case 86:
            var9 = this.readType();
            var10 = this.readInt();
            var12 = this.findSerializerFactory().getListDeserializer(var9, cl);
            v1 = var12.readLengthList(this, var10);
            return v1;
         case 87:
            var7 = this.findSerializerFactory().getListDeserializer((String)null, cl);
            Object var11 = var7.readList(this, -1);
            return var11;
         case 88:
            value = this.readInt();
            reader = this.findSerializerFactory().getListDeserializer((String)null, cl);
            v = reader.readLengthList(this, value);
            return v;
         case 96:
         case 97:
         case 98:
         case 99:
         case 100:
         case 101:
         case 102:
         case 103:
         case 104:
         case 105:
         case 106:
         case 107:
         case 108:
         case 109:
         case 110:
         case 111:
            value = tag - 96;
            var10 = this._classDefs.size();
            if(value >= 0 && var10 > value) {
               var13 = (Hessian2Input.ObjectDefinition)this._classDefs.get(value);
               return this.readObjectInstance(cl, var13);
            }

            throw new HessianProtocolException("\'" + value + "\' is an unknown class definition");
         case 112:
         case 113:
         case 114:
         case 115:
         case 116:
         case 117:
         case 118:
         case 119:
            value = tag - 112;
            String var8 = this.readType();
            var12 = this.findSerializerFactory().getListDeserializer(var8, cl);
            v1 = var12.readLengthList(this, value);
            return v1;
         case 120:
         case 121:
         case 122:
         case 123:
         case 124:
         case 125:
         case 126:
         case 127:
            value = tag - 120;
            reader = this.findSerializerFactory().getListDeserializer((String)null, cl);
            v = reader.readLengthList(this, value);
            return v;
         }
      } else {
         return this.readObject();
      }
   }

   public Object readObject() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int tag = var10000;
      int ref;
      Deserializer def;
      Deserializer reader;
      String var5;
      int var6;
      Hessian2Input.ObjectDefinition var7;
      int var8;
      byte[] var10;
      switch(tag) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      case 16:
      case 17:
      case 18:
      case 19:
      case 20:
      case 21:
      case 22:
      case 23:
      case 24:
      case 25:
      case 26:
      case 27:
      case 28:
      case 29:
      case 30:
      case 31:
         this._isLastChunk = true;
         this._chunkLength = tag - 0;
         this._sbuf.setLength(0);
         this.parseString(this._sbuf);
         return this._sbuf.toString();
      case 32:
      case 33:
      case 34:
      case 35:
      case 36:
      case 37:
      case 38:
      case 39:
      case 40:
      case 41:
      case 42:
      case 43:
      case 44:
      case 45:
      case 46:
      case 47:
         this._isLastChunk = true;
         ref = tag - 32;
         this._chunkLength = 0;
         var10 = new byte[ref];

         for(var8 = 0; var8 < ref; ++var8) {
            var10[var8] = (byte)this.read();
         }

         return var10;
      case 48:
      case 49:
      case 50:
      case 51:
         this._isLastChunk = true;
         this._chunkLength = (tag - 48) * 256 + this.read();
         this._sbuf.setLength(0);
         this.parseString(this._sbuf);
         return this._sbuf.toString();
      case 52:
      case 53:
      case 54:
      case 55:
         this._isLastChunk = true;
         ref = (tag - 52) * 256 + this.read();
         this._chunkLength = 0;
         var10 = new byte[ref];

         for(var8 = 0; var8 < ref; ++var8) {
            var10[var8] = (byte)this.read();
         }

         return var10;
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
         return Long.valueOf((long)((tag - 60 << 16) + 256 * this.read() + this.read()));
      case 64:
      case 69:
      case 71:
      case 80:
      case 90:
      default:
         if(tag < 0) {
            throw new EOFException("readObject: unexpected end of file");
         }

         throw this.error("readObject: unknown code " + this.codeName(tag));
      case 65:
      case 66:
         this._isLastChunk = tag == 66;
         this._chunkLength = (this.read() << 8) + this.read();
         ByteArrayOutputStream var9 = new ByteArrayOutputStream();

         while((ref = this.parseByte()) >= 0) {
            var9.write(ref);
         }

         return var9.toByteArray();
      case 67:
         this.readObjectDefinition((Class)null);
         return this.readObject();
      case 68:
         return Double.valueOf(this.parseDouble());
      case 70:
         return Boolean.valueOf(false);
      case 72:
         return this.findSerializerFactory().readMap(this, (String)null);
      case 73:
         return Integer.valueOf(this.parseInt());
      case 74:
         return new Date(this.parseLong());
      case 75:
         return new Date((long)this.parseInt() * 60000L);
      case 76:
         return Long.valueOf(this.parseLong());
      case 77:
         var5 = this.readType();
         return this.findSerializerFactory().readMap(this, var5);
      case 78:
         return null;
      case 79:
         ref = this.readInt();
         if(this._classDefs.size() <= ref) {
            throw this.error("Illegal object reference #" + ref);
         }

         var7 = (Hessian2Input.ObjectDefinition)this._classDefs.get(ref);
         return this.readObjectInstance((Class)null, var7);
      case 81:
         ref = this.readInt();
         return this._refs.get(ref);
      case 82:
      case 83:
         this._isLastChunk = tag == 83;
         this._chunkLength = (this.read() << 8) + this.read();
         this._sbuf.setLength(0);
         this.parseString(this._sbuf);
         return this._sbuf.toString();
      case 84:
         return Boolean.valueOf(true);
      case 85:
         var5 = this.readType();
         return this.findSerializerFactory().readList(this, -1, var5);
      case 86:
         var5 = this.readType();
         var6 = this.readInt();
         reader = this.findSerializerFactory().getListDeserializer(var5, (Class)null);
         return reader.readLengthList(this, var6);
      case 87:
         return this.findSerializerFactory().readList(this, -1, (String)null);
      case 88:
         ref = this.readInt();
         def = this.findSerializerFactory().getListDeserializer((String)null, (Class)null);
         return def.readLengthList(this, ref);
      case 89:
         return Long.valueOf((long)this.parseInt());
      case 91:
         return Double.valueOf(0.0D);
      case 92:
         return Double.valueOf(1.0D);
      case 93:
         return Double.valueOf((double)((byte)this.read()));
      case 94:
         return Double.valueOf((double)((short)(256 * this.read() + this.read())));
      case 95:
         ref = this.parseInt();
         return Double.valueOf(0.001D * (double)ref);
      case 96:
      case 97:
      case 98:
      case 99:
      case 100:
      case 101:
      case 102:
      case 103:
      case 104:
      case 105:
      case 106:
      case 107:
      case 108:
      case 109:
      case 110:
      case 111:
         ref = tag - 96;
         if(this._classDefs.size() <= ref) {
            throw this.error("No classes defined at reference \'" + Integer.toHexString(tag) + "\'");
         }

         var7 = (Hessian2Input.ObjectDefinition)this._classDefs.get(ref);
         return this.readObjectInstance((Class)null, var7);
      case 112:
      case 113:
      case 114:
      case 115:
      case 116:
      case 117:
      case 118:
      case 119:
         var5 = this.readType();
         var6 = tag - 112;
         reader = this.findSerializerFactory().getListDeserializer(var5, (Class)null);
         return reader.readLengthList(this, var6);
      case 120:
      case 121:
      case 122:
      case 123:
      case 124:
      case 125:
      case 126:
      case 127:
         ref = tag - 120;
         def = this.findSerializerFactory().getListDeserializer((String)null, (Class)null);
         return def.readLengthList(this, ref);
      case 128:
      case 129:
      case 130:
      case 131:
      case 132:
      case 133:
      case 134:
      case 135:
      case 136:
      case 137:
      case 138:
      case 139:
      case 140:
      case 141:
      case 142:
      case 143:
      case 144:
      case 145:
      case 146:
      case 147:
      case 148:
      case 149:
      case 150:
      case 151:
      case 152:
      case 153:
      case 154:
      case 155:
      case 156:
      case 157:
      case 158:
      case 159:
      case 160:
      case 161:
      case 162:
      case 163:
      case 164:
      case 165:
      case 166:
      case 167:
      case 168:
      case 169:
      case 170:
      case 171:
      case 172:
      case 173:
      case 174:
      case 175:
      case 176:
      case 177:
      case 178:
      case 179:
      case 180:
      case 181:
      case 182:
      case 183:
      case 184:
      case 185:
      case 186:
      case 187:
      case 188:
      case 189:
      case 190:
      case 191:
         return Integer.valueOf(tag - 144);
      case 192:
      case 193:
      case 194:
      case 195:
      case 196:
      case 197:
      case 198:
      case 199:
      case 200:
      case 201:
      case 202:
      case 203:
      case 204:
      case 205:
      case 206:
      case 207:
         return Integer.valueOf((tag - 200 << 8) + this.read());
      case 208:
      case 209:
      case 210:
      case 211:
      case 212:
      case 213:
      case 214:
      case 215:
         return Integer.valueOf((tag - 212 << 16) + 256 * this.read() + this.read());
      case 216:
      case 217:
      case 218:
      case 219:
      case 220:
      case 221:
      case 222:
      case 223:
      case 224:
      case 225:
      case 226:
      case 227:
      case 228:
      case 229:
      case 230:
      case 231:
      case 232:
      case 233:
      case 234:
      case 235:
      case 236:
      case 237:
      case 238:
      case 239:
         return Long.valueOf((long)(tag - 224));
      case 240:
      case 241:
      case 242:
      case 243:
      case 244:
      case 245:
      case 246:
      case 247:
      case 248:
      case 249:
      case 250:
      case 251:
      case 252:
      case 253:
      case 254:
      case 255:
         return Long.valueOf((long)((tag - 248 << 8) + this.read()));
      }
   }

   private void readObjectDefinition(Class cl) throws IOException {
      String type = this.readString();
      int len = this.readInt();
      SerializerFactory factory = this.findSerializerFactory();
      Deserializer reader = factory.getObjectDeserializer(type, (Class)null);
      Object[] fields = reader.createFields(len);
      String[] fieldNames = new String[len];

      for(int def = 0; def < len; ++def) {
         String name = this.readString();
         fields[def] = reader.createField(name);
         fieldNames[def] = name;
      }

      Hessian2Input.ObjectDefinition var10 = new Hessian2Input.ObjectDefinition(type, reader, fields, fieldNames);
      this._classDefs.add(var10);
   }

   private Object readObjectInstance(Class cl, Hessian2Input.ObjectDefinition def) throws IOException {
      String type = def.getType();
      Deserializer reader = def.getReader();
      Object[] fields = def.getFields();
      SerializerFactory factory = this.findSerializerFactory();
      if(cl != reader.getType() && cl != null) {
         reader = factory.getObjectDeserializer(type, cl);
         return reader.readObject(this, (String[])def.getFieldNames());
      } else {
         return reader.readObject(this, (Object[])fields);
      }
   }

   public Object readRemote() throws IOException {
      String type = this.readType();
      String url = this.readString();
      return this.resolveRemote(type, url);
   }

   public Object readRef() throws IOException {
      int value = this.parseInt();
      return this._refs.get(value);
   }

   public int readListStart() throws IOException {
      return this.read();
   }

   public int readMapStart() throws IOException {
      return this.read();
   }

   public boolean isEnd() throws IOException {
      int code;
      if(this._offset < this._length) {
         code = this._buffer[this._offset] & 255;
      } else {
         code = this.read();
         if(code >= 0) {
            --this._offset;
         }
      }

      return code < 0 || code == 90;
   }

   public void readEnd() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int code = var10000;
      if(code != 90) {
         if(code < 0) {
            throw this.error("unexpected end of file");
         } else {
            throw this.error("unknown code:" + this.codeName(code));
         }
      }
   }

   public void readMapEnd() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int code = var10000;
      if(code != 90) {
         throw this.error("expected end of map (\'Z\') at \'" + this.codeName(code) + "\'");
      }
   }

   public void readListEnd() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int code = var10000;
      if(code != 90) {
         throw this.error("expected end of list (\'Z\') at \'" + this.codeName(code) + "\'");
      }
   }

   public int addRef(Object ref) {
      if(this._refs == null) {
         this._refs = new ArrayList();
      }

      this._refs.add(ref);
      return this._refs.size() - 1;
   }

   public void setRef(int i, Object ref) {
      this._refs.set(i, ref);
   }

   public void resetReferences() {
      this._refs.clear();
   }

   public void reset() {
      this.resetReferences();
      this._classDefs.clear();
      this._types.clear();
   }

   public void resetBuffer() {
      int offset = this._offset;
      this._offset = 0;
      int length = this._length;
      this._length = 0;
      if(length > 0 && offset != length) {
         throw new IllegalStateException("offset=" + offset + " length=" + length);
      }
   }

   public Object readStreamingObject() throws IOException {
      if(this._refs != null) {
         this._refs.clear();
      }

      return this.readObject();
   }

   public Object resolveRemote(String type, String url) throws IOException {
      HessianRemoteResolver resolver = this.getRemoteResolver();
      return resolver != null?resolver.lookup(type, url):new HessianRemote(type, url);
   }

   public String readType() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int code = var10000;
      --this._offset;
      switch(code) {
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
      case 9:
      case 10:
      case 11:
      case 12:
      case 13:
      case 14:
      case 15:
      case 16:
      case 17:
      case 18:
      case 19:
      case 20:
      case 21:
      case 22:
      case 23:
      case 24:
      case 25:
      case 26:
      case 27:
      case 28:
      case 29:
      case 30:
      case 31:
      case 48:
      case 49:
      case 50:
      case 51:
      case 82:
      case 83:
         String ref = this.readString();
         if(this._types == null) {
            this._types = new ArrayList();
         }

         this._types.add(ref);
         return ref;
      case 32:
      case 33:
      case 34:
      case 35:
      case 36:
      case 37:
      case 38:
      case 39:
      case 40:
      case 41:
      case 42:
      case 43:
      case 44:
      case 45:
      case 46:
      case 47:
      case 52:
      case 53:
      case 54:
      case 55:
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
      case 64:
      case 65:
      case 66:
      case 67:
      case 68:
      case 69:
      case 70:
      case 71:
      case 72:
      case 73:
      case 74:
      case 75:
      case 76:
      case 77:
      case 78:
      case 79:
      case 80:
      case 81:
      default:
         int var3 = this.readInt();
         if(this._types.size() <= var3) {
            throw new IndexOutOfBoundsException("type ref #" + var3 + " is greater than the number of valid types (" + this._types.size() + ")");
         } else {
            return (String)this._types.get(var3);
         }
      }
   }

   public int readLength() throws IOException {
      throw new UnsupportedOperationException();
   }

   private int parseInt() throws IOException {
      int offset = this._offset;
      int b24;
      int b16;
      int b8;
      if(offset + 3 < this._length) {
         byte[] b321 = this._buffer;
         b24 = b321[offset + 0] & 255;
         b16 = b321[offset + 1] & 255;
         b8 = b321[offset + 2] & 255;
         int b81 = b321[offset + 3] & 255;
         this._offset = offset + 4;
         return (b24 << 24) + (b16 << 16) + (b8 << 8) + b81;
      } else {
         int b32 = this.read();
         b24 = this.read();
         b16 = this.read();
         b8 = this.read();
         return (b32 << 24) + (b24 << 16) + (b16 << 8) + b8;
      }
   }

   private long parseLong() throws IOException {
      long b64 = (long)this.read();
      long b56 = (long)this.read();
      long b48 = (long)this.read();
      long b40 = (long)this.read();
      long b32 = (long)this.read();
      long b24 = (long)this.read();
      long b16 = (long)this.read();
      long b8 = (long)this.read();
      return (b64 << 56) + (b56 << 48) + (b48 << 40) + (b40 << 32) + (b32 << 24) + (b24 << 16) + (b16 << 8) + b8;
   }

   private double parseDouble() throws IOException {
      long bits = this.parseLong();
      return Double.longBitsToDouble(bits);
   }

   Node parseXML() throws IOException {
      throw new UnsupportedOperationException();
   }

   private void parseString(StringBuilder sbuf) throws IOException {
      while(this._chunkLength > 0 || this.parseChunkLength()) {
         int length = this._chunkLength;
         this._chunkLength = 0;

         while(length-- > 0) {
            sbuf.append((char)this.parseUTF8Char());
         }
      }

   }

   private int parseChar() throws IOException {
      while(true) {
         if(this._chunkLength <= 0) {
            if(this.parseChunkLength()) {
               continue;
            }

            return -1;
         }

         --this._chunkLength;
         return this.parseUTF8Char();
      }
   }

   private boolean parseChunkLength() throws IOException {
      if(this._isLastChunk) {
         return false;
      } else {
         int var10000;
         if(this._offset < this._length) {
            ++this._offset;
            var10000 = this._buffer[this._offset] & 255;
         } else {
            var10000 = this.read();
         }

         int code = var10000;
         switch(code) {
         case 0:
         case 1:
         case 2:
         case 3:
         case 4:
         case 5:
         case 6:
         case 7:
         case 8:
         case 9:
         case 10:
         case 11:
         case 12:
         case 13:
         case 14:
         case 15:
         case 16:
         case 17:
         case 18:
         case 19:
         case 20:
         case 21:
         case 22:
         case 23:
         case 24:
         case 25:
         case 26:
         case 27:
         case 28:
         case 29:
         case 30:
         case 31:
            this._isLastChunk = true;
            this._chunkLength = code - 0;
            break;
         case 32:
         case 33:
         case 34:
         case 35:
         case 36:
         case 37:
         case 38:
         case 39:
         case 40:
         case 41:
         case 42:
         case 43:
         case 44:
         case 45:
         case 46:
         case 47:
         case 52:
         case 53:
         case 54:
         case 55:
         case 56:
         case 57:
         case 58:
         case 59:
         case 60:
         case 61:
         case 62:
         case 63:
         case 64:
         case 65:
         case 66:
         case 67:
         case 68:
         case 69:
         case 70:
         case 71:
         case 72:
         case 73:
         case 74:
         case 75:
         case 76:
         case 77:
         case 78:
         case 79:
         case 80:
         case 81:
         default:
            throw this.expect("string", code);
         case 48:
         case 49:
         case 50:
         case 51:
            this._isLastChunk = true;
            this._chunkLength = (code - 48) * 256 + this.read();
            break;
         case 82:
            this._isLastChunk = false;
            this._chunkLength = (this.read() << 8) + this.read();
            break;
         case 83:
            this._isLastChunk = true;
            this._chunkLength = (this.read() << 8) + this.read();
         }

         return true;
      }
   }

   private int parseUTF8Char() throws IOException {
      int var10000;
      if(this._offset < this._length) {
         ++this._offset;
         var10000 = this._buffer[this._offset] & 255;
      } else {
         var10000 = this.read();
      }

      int ch = var10000;
      if(ch < 128) {
         return ch;
      } else {
         int ch1;
         int ch2;
         if((ch & 224) == 192) {
            ch1 = this.read();
            ch2 = ((ch & 31) << 6) + (ch1 & 63);
            return ch2;
         } else if((ch & 240) == 224) {
            ch1 = this.read();
            ch2 = this.read();
            int v = ((ch & 15) << 12) + ((ch1 & 63) << 6) + (ch2 & 63);
            return v;
         } else {
            throw this.error("bad utf-8 encoding at " + this.codeName(ch));
         }
      }
   }

   private int parseByte() throws IOException {
      while(this._chunkLength <= 0) {
         if(this._isLastChunk) {
            return -1;
         }

         int code = this.read();
         switch(code) {
         case 32:
         case 33:
         case 34:
         case 35:
         case 36:
         case 37:
         case 38:
         case 39:
         case 40:
         case 41:
         case 42:
         case 43:
         case 44:
         case 45:
         case 46:
         case 47:
            this._isLastChunk = true;
            this._chunkLength = code - 32;
            break;
         case 48:
         case 49:
         case 50:
         case 51:
         case 56:
         case 57:
         case 58:
         case 59:
         case 60:
         case 61:
         case 62:
         case 63:
         case 64:
         default:
            throw this.expect("byte[]", code);
         case 52:
         case 53:
         case 54:
         case 55:
            this._isLastChunk = true;
            this._chunkLength = (code - 52) * 256 + this.read();
            break;
         case 65:
            this._isLastChunk = false;
            this._chunkLength = (this.read() << 8) + this.read();
            break;
         case 66:
            this._isLastChunk = true;
            this._chunkLength = (this.read() << 8) + this.read();
         }
      }

      --this._chunkLength;
      return this.read();
   }

   public InputStream readInputStream() throws IOException {
      int tag = this.read();
      switch(tag) {
      case 32:
      case 33:
      case 34:
      case 35:
      case 36:
      case 37:
      case 38:
      case 39:
      case 40:
      case 41:
      case 42:
      case 43:
      case 44:
      case 45:
      case 46:
      case 47:
         this._isLastChunk = true;
         this._chunkLength = tag - 32;
         break;
      case 48:
      case 49:
      case 50:
      case 51:
      case 56:
      case 57:
      case 58:
      case 59:
      case 60:
      case 61:
      case 62:
      case 63:
      case 64:
      case 67:
      case 68:
      case 69:
      case 70:
      case 71:
      case 72:
      case 73:
      case 74:
      case 75:
      case 76:
      case 77:
      default:
         throw this.expect("binary", tag);
      case 52:
      case 53:
      case 54:
      case 55:
         this._isLastChunk = true;
         this._chunkLength = (tag - 52) * 256 + this.read();
         break;
      case 65:
      case 66:
         this._isLastChunk = tag == 66;
         this._chunkLength = (this.read() << 8) + this.read();
         break;
      case 78:
         return null;
      }

      return new Hessian2Input.ReadInputStream();
   }

   int read(byte[] buffer, int offset, int length) throws IOException {
      int readLength;
      int sublen;
      for(readLength = 0; length > 0; this._chunkLength -= sublen) {
         while(this._chunkLength <= 0) {
            if(this._isLastChunk) {
               return readLength == 0?-1:readLength;
            }

            sublen = this.read();
            switch(sublen) {
            case 32:
            case 33:
            case 34:
            case 35:
            case 36:
            case 37:
            case 38:
            case 39:
            case 40:
            case 41:
            case 42:
            case 43:
            case 44:
            case 45:
            case 46:
            case 47:
               this._isLastChunk = true;
               this._chunkLength = sublen - 32;
               break;
            case 48:
            case 49:
            case 50:
            case 51:
            case 56:
            case 57:
            case 58:
            case 59:
            case 60:
            case 61:
            case 62:
            case 63:
            case 64:
            default:
               throw this.expect("byte[]", sublen);
            case 52:
            case 53:
            case 54:
            case 55:
               this._isLastChunk = true;
               this._chunkLength = (sublen - 52) * 256 + this.read();
               break;
            case 65:
               this._isLastChunk = false;
               this._chunkLength = (this.read() << 8) + this.read();
               break;
            case 66:
               this._isLastChunk = true;
               this._chunkLength = (this.read() << 8) + this.read();
            }
         }

         sublen = this._chunkLength;
         if(length < sublen) {
            sublen = length;
         }

         if(this._length <= this._offset && !this.readBuffer()) {
            return -1;
         }

         if(this._length - this._offset < sublen) {
            sublen = this._length - this._offset;
         }

         System.arraycopy(this._buffer, this._offset, buffer, offset, sublen);
         this._offset += sublen;
         offset += sublen;
         readLength += sublen;
         length -= sublen;
      }

      return readLength;
   }

   public final int read() throws IOException {
      if(this._length <= this._offset && !this.readBuffer()) {
         return -1;
      } else {
         ++this._offset;
         return this._buffer[this._offset] & 255;
      }
   }

   protected void unread() {
      if(this._offset <= 0) {
         throw new IllegalStateException();
      } else {
         --this._offset;
      }
   }

   private final boolean readBuffer() throws IOException {
      byte[] buffer = this._buffer;
      int offset = this._offset;
      int length = this._length;
      if(offset < length) {
         System.arraycopy(buffer, offset, buffer, 0, length - offset);
         offset = length - offset;
      } else {
         offset = 0;
      }

      int len = this._is.read(buffer, offset, 1024 - offset);
      if(len <= 0) {
         this._length = offset;
         this._offset = 0;
         return offset > 0;
      } else {
         this._length = offset + len;
         this._offset = 0;
         return true;
      }
   }

   public Reader getReader() {
      return null;
   }

   protected IOException expect(String expect, int ch) throws IOException {
      if(ch < 0) {
         return this.error("expected " + expect + " at end of file");
      } else {
         --this._offset;

         try {
            int e = this._offset;
            String context = this.buildDebugContext(this._buffer, 0, this._length, e);
            Object obj = this.readObject();
            return obj != null?this.error("expected " + expect + " at 0x" + Integer.toHexString(ch & 255) + " " + obj.getClass().getName() + " (" + obj + ")" + "\n  " + context + ""):this.error("expected " + expect + " at 0x" + Integer.toHexString(ch & 255) + " null");
         } catch (Exception var6) {
            log.log(Level.FINE, var6.toString(), var6);
            return this.error("expected " + expect + " at 0x" + Integer.toHexString(ch & 255));
         }
      }
   }

   private String buildDebugContext(byte[] buffer, int offset, int length, int errorOffset) {
      StringBuilder sb = new StringBuilder();
      sb.append("[");

      int i;
      byte ch;
      for(i = 0; i < errorOffset; ++i) {
         ch = buffer[offset + i];
         this.addDebugChar(sb, ch);
      }

      sb.append("] ");
      this.addDebugChar(sb, buffer[offset + errorOffset]);
      sb.append(" [");

      for(i = errorOffset + 1; i < length; ++i) {
         ch = buffer[offset + i];
         this.addDebugChar(sb, ch);
      }

      sb.append("]");
      return sb.toString();
   }

   private void addDebugChar(StringBuilder sb, int ch) {
      if(ch >= 32 && ch < 127) {
         sb.append((char)ch);
      } else if(ch == 10) {
         sb.append((char)ch);
      } else {
         sb.append(String.format("\\x%02x", new Object[]{Integer.valueOf(ch & 255)}));
      }

   }

   protected String codeName(int ch) {
      return ch < 0?"end of file":"0x" + Integer.toHexString(ch & 255) + " (" + (char)ch + ")";
   }

   protected IOException error(String message) {
      return this._method != null?new HessianProtocolException(this._method + ": " + message):new HessianProtocolException(message);
   }

   public void free() {
      this.reset();
   }

   public void close() throws IOException {
      InputStream is = this._is;
      this._is = null;
      if(_isCloseStreamOnClose && is != null) {
         is.close();
      }

   }

   static {
      try {
         _detailMessageField = Throwable.class.getDeclaredField("detailMessage");
         _detailMessageField.setAccessible(true);
      } catch (Throwable var1) {
         ;
      }

   }

   static final class ObjectDefinition {

      private final String _type;
      private final Deserializer _reader;
      private final Object[] _fields;
      private final String[] _fieldNames;


      ObjectDefinition(String type, Deserializer reader, Object[] fields, String[] fieldNames) {
         this._type = type;
         this._reader = reader;
         this._fields = fields;
         this._fieldNames = fieldNames;
      }

      String getType() {
         return this._type;
      }

      Deserializer getReader() {
         return this._reader;
      }

      Object[] getFields() {
         return this._fields;
      }

      String[] getFieldNames() {
         return this._fieldNames;
      }
   }

   class ReadInputStream extends InputStream {

      boolean _isClosed = false;


      public int read() throws IOException {
         if(this._isClosed) {
            return -1;
         } else {
            int ch = Hessian2Input.this.parseByte();
            if(ch < 0) {
               this._isClosed = true;
            }

            return ch;
         }
      }

      public int read(byte[] buffer, int offset, int length) throws IOException {
         if(this._isClosed) {
            return -1;
         } else {
            int len = Hessian2Input.this.read(buffer, offset, length);
            if(len < 0) {
               this._isClosed = true;
            }

            return len;
         }
      }

      public void close() throws IOException {
         while(this.read() >= 0) {
            ;
         }

      }
   }
}
