package com.caucho.burlap.client;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.TimeZone;
import java.util.Vector;

public class MicroBurlapOutput {

   private OutputStream os;
   private Date date;
   private Calendar utcCalendar;
   private Calendar localCalendar;


   public MicroBurlapOutput(OutputStream os) {
      this.init(os);
   }

   public MicroBurlapOutput() {
   }

   public void init(OutputStream os) {
      this.os = os;
   }

   public void call(String method, Object[] args) throws IOException {
      this.startCall(method);
      if(args != null) {
         for(int i = 0; i < args.length; ++i) {
            this.writeObject(args[i]);
         }
      }

      this.completeCall();
   }

   public void startCall(String method) throws IOException {
      this.print("<burlap:call><method>");
      this.print(method);
      this.print("</method>");
   }

   public void completeCall() throws IOException {
      this.print("</burlap:call>");
   }

   public void writeBoolean(boolean value) throws IOException {
      this.print("<boolean>");
      this.printInt(value?1:0);
      this.print("</boolean>");
   }

   public void writeInt(int value) throws IOException {
      this.print("<int>");
      this.printInt(value);
      this.print("</int>");
   }

   public void writeLong(long value) throws IOException {
      this.print("<long>");
      this.printLong(value);
      this.print("</long>");
   }

   public void writeNull() throws IOException {
      this.print("<null></null>");
   }

   public void writeString(String value) throws IOException {
      if(value == null) {
         this.print("<null></null>");
      } else {
         this.print("<string>");
         this.printString(value);
         this.print("</string>");
      }

   }

   public void writeBytes(byte[] buffer, int offset, int length) throws IOException {
      if(buffer == null) {
         this.print("<null></null>");
      } else {
         this.print("<base64>");
         this.printBytes(buffer, offset, length);
         this.print("</base64>");
      }

   }

   public void writeUTCDate(long time) throws IOException {
      this.print("<date>");
      if(this.utcCalendar == null) {
         this.utcCalendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
         this.date = new Date();
      }

      this.date.setTime(time);
      this.utcCalendar.setTime(this.date);
      this.printDate(this.utcCalendar);
      this.print("</date>");
   }

   public void writeLocalDate(long time) throws IOException {
      this.print("<date>");
      if(this.localCalendar == null) {
         this.localCalendar = Calendar.getInstance();
         this.date = new Date();
      }

      this.date.setTime(time);
      this.localCalendar.setTime(this.date);
      this.printDate(this.localCalendar);
      this.print("</date>");
   }

   public void writeRef(int value) throws IOException {
      this.print("<ref>");
      this.printInt(value);
      this.print("</ref>");
   }

   public void writeObject(Object object) throws IOException {
      if(object == null) {
         this.writeNull();
      } else if(object instanceof String) {
         this.writeString((String)object);
      } else if(object instanceof Boolean) {
         this.writeBoolean(((Boolean)object).booleanValue());
      } else if(object instanceof Integer) {
         this.writeInt(((Integer)object).intValue());
      } else if(object instanceof Long) {
         this.writeLong(((Long)object).longValue());
      } else if(object instanceof Date) {
         this.writeUTCDate(((Date)object).getTime());
      } else if(object instanceof byte[]) {
         byte[] hashtable = (byte[])((byte[])object);
         this.writeBytes(hashtable, 0, hashtable.length);
      } else if(object instanceof Vector) {
         Vector var6 = (Vector)object;
         int e = var6.size();
         this.writeListBegin(e, (String)null);

         for(int key = 0; key < e; ++key) {
            this.writeObject(var6.elementAt(key));
         }

         this.writeListEnd();
      } else if(object instanceof Hashtable) {
         Hashtable var7 = (Hashtable)object;
         this.writeMapBegin((String)null);
         Enumeration var8 = var7.keys();

         while(var8.hasMoreElements()) {
            Object var9 = var8.nextElement();
            Object value = var7.get(var9);
            this.writeObject(var9);
            this.writeObject(value);
         }

         this.writeMapEnd();
      } else {
         this.writeCustomObject(object);
      }

   }

   public void writeCustomObject(Object object) throws IOException {
      throw new IOException("unexpected object: " + object);
   }

   public void writeListBegin(int length, String type) throws IOException {
      this.print("<list><type>");
      if(type != null) {
         this.print(type);
      }

      this.print("</type><length>");
      this.printInt(length);
      this.print("</length>");
   }

   public void writeListEnd() throws IOException {
      this.print("</list>");
   }

   public void writeMapBegin(String type) throws IOException {
      this.print("<map><type>");
      if(type != null) {
         this.print(type);
      }

      this.print("</type>");
   }

   public void writeMapEnd() throws IOException {
      this.print("</map>");
   }

   public void writeRemote(String type, String url) throws IOException {
      this.print("<remote><type>");
      if(type != null) {
         this.print(type);
      }

      this.print("</type><string>");
      this.print(url);
      this.print("</string></remote>");
   }

   public void printInt(int v) throws IOException {
      this.print(String.valueOf(v));
   }

   public void printLong(long v) throws IOException {
      this.print(String.valueOf(v));
   }

   public void printString(String v) throws IOException {
      int len = v.length();

      for(int i = 0; i < len; ++i) {
         char ch = v.charAt(i);
         switch(ch) {
         case 13:
            this.print("&#13;");
            break;
         case 38:
            this.print("&amp;");
            break;
         case 60:
            this.print("&lt;");
            break;
         default:
            if(ch < 128) {
               this.os.write(ch);
            } else if(ch < 2048) {
               this.os.write(192 + (ch >> 6 & 31));
               this.os.write(128 + (ch & 63));
            } else {
               this.os.write(224 + (ch >> 12 & 15));
               this.os.write(128 + (ch >> 6 & 63));
               this.os.write(128 + (ch & 63));
            }
         }
      }

   }

   public void printBytes(byte[] data, int offset, int length) throws IOException {
      int chunk;
      while(length >= 3) {
         chunk = ((data[offset] & 255) << 16) + ((data[offset + 1] & 255) << 8) + (data[offset + 2] & 255);
         this.os.write(base64encode(chunk >> 18));
         this.os.write(base64encode(chunk >> 12));
         this.os.write(base64encode(chunk >> 6));
         this.os.write(base64encode(chunk));
         offset += 3;
         length -= 3;
      }

      if(length == 2) {
         chunk = ((data[offset] & 255) << 8) + (data[offset + 1] & 255);
         this.os.write(base64encode(chunk >> 12));
         this.os.write(base64encode(chunk >> 6));
         this.os.write(base64encode(chunk));
         this.os.write(61);
      } else if(length == 1) {
         chunk = data[offset] & 255;
         this.os.write(base64encode(chunk >> 6));
         this.os.write(base64encode(chunk));
         this.os.write(61);
         this.os.write(61);
      }

   }

   public static char base64encode(int d) {
      d &= 63;
      return (char)(d < 26?(char)(d + 65):(d < 52?(char)(d + 97 - 26):(d < 62?(char)(d + 48 - 52):(d == 62?'+':'/'))));
   }

   public void printDate(Calendar calendar) throws IOException {
      int year = calendar.get(1);
      this.os.write((char)(48 + year / 1000 % 10));
      this.os.write((char)(48 + year / 100 % 10));
      this.os.write((char)(48 + year / 10 % 10));
      this.os.write((char)(48 + year % 10));
      int month = calendar.get(2) + 1;
      this.os.write((char)(48 + month / 10 % 10));
      this.os.write((char)(48 + month % 10));
      int day = calendar.get(5);
      this.os.write((char)(48 + day / 10 % 10));
      this.os.write((char)(48 + day % 10));
      this.os.write(84);
      int hour = calendar.get(11);
      this.os.write((char)(48 + hour / 10 % 10));
      this.os.write((char)(48 + hour % 10));
      int minute = calendar.get(12);
      this.os.write((char)(48 + minute / 10 % 10));
      this.os.write((char)(48 + minute % 10));
      int second = calendar.get(13);
      this.os.write((char)(48 + second / 10 % 10));
      this.os.write((char)(48 + second % 10));
      this.os.write(90);
   }

   public void print(String s) throws IOException {
      int len = s.length();

      for(int i = 0; i < len; ++i) {
         char ch = s.charAt(i);
         this.os.write(ch);
      }

   }
}
