package com.caucho.burlap.client;

import com.caucho.burlap.client.BurlapProtocolException;
import com.caucho.burlap.client.BurlapRemote;
import com.caucho.burlap.client.BurlapServiceException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Calendar;
import java.util.Date;
import java.util.Hashtable;
import java.util.TimeZone;
import java.util.Vector;

public class MicroBurlapInput {

   private static int[] base64Decode = new int[256];
   private InputStream is;
   protected int peek;
   protected boolean peekTag;
   protected Date date;
   protected Calendar utcCalendar;
   private Calendar localCalendar;
   protected Vector refs;
   protected String method;
   protected StringBuffer sbuf = new StringBuffer();
   protected StringBuffer entity = new StringBuffer();


   public MicroBurlapInput(InputStream is) {
      this.init(is);
   }

   public MicroBurlapInput() {
   }

   public String getMethod() {
      return this.method;
   }

   public void init(InputStream is) {
      this.is = is;
      this.refs = null;
   }

   public void startCall() throws IOException {
      this.expectStartTag("burlap:call");
      this.expectStartTag("method");
      this.method = this.parseString();
      this.expectEndTag("method");
      this.refs = null;
   }

   public void completeCall() throws IOException {
      this.expectEndTag("burlap:call");
   }

   public Object readReply(Class expectedClass) throws Exception {
      if(this.startReply()) {
         Object fault1 = this.readObject(expectedClass);
         this.completeReply();
         return fault1;
      } else {
         Hashtable fault = this.readFault();
         Object detail = fault.get("detail");
         if(detail instanceof Exception) {
            throw (Exception)detail;
         } else {
            String code = (String)fault.get("code");
            String message = (String)fault.get("message");
            throw new BurlapServiceException(message, code, detail);
         }
      }
   }

   public boolean startReply() throws IOException {
      this.refs = null;
      this.expectStartTag("burlap:reply");
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <value>");
      } else {
         String tag = this.sbuf.toString();
         if(tag.equals("fault")) {
            this.peekTag = true;
            return false;
         } else {
            this.peekTag = true;
            return true;
         }
      }
   }

   public void completeReply() throws IOException {
      this.expectEndTag("burlap:reply");
   }

   public boolean readBoolean() throws IOException {
      this.expectStartTag("boolean");
      int value = this.parseInt();
      this.expectEndTag("boolean");
      return value != 0;
   }

   public int readInt() throws IOException {
      this.expectStartTag("int");
      int value = this.parseInt();
      this.expectEndTag("int");
      return value;
   }

   public long readLong() throws IOException {
      this.expectStartTag("long");
      long value = this.parseLong();
      this.expectEndTag("long");
      return value;
   }

   public long readUTCDate() throws IOException {
      this.expectStartTag("date");
      if(this.utcCalendar == null) {
         this.utcCalendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
      }

      long value = this.parseDate(this.utcCalendar);
      this.expectEndTag("date");
      return value;
   }

   public long readLocalDate() throws IOException {
      this.expectStartTag("date");
      if(this.localCalendar == null) {
         this.localCalendar = Calendar.getInstance();
      }

      long value = this.parseDate(this.localCalendar);
      this.expectEndTag("date");
      return value;
   }

   public BurlapRemote readRemote() throws IOException {
      this.expectStartTag("remote");
      String type = this.readType();
      String url = this.readString();
      this.expectEndTag("remote");
      return new BurlapRemote(type, url);
   }

   public String readString() throws IOException {
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <string>");
      } else {
         String tag = this.sbuf.toString();
         if(tag.equals("null")) {
            this.expectEndTag("null");
            return null;
         } else if(tag.equals("string")) {
            this.sbuf.setLength(0);
            this.parseString(this.sbuf);
            String value = this.sbuf.toString();
            this.expectEndTag("string");
            return value;
         } else {
            throw this.expectBeginTag("string", tag);
         }
      }
   }

   public byte[] readBytes() throws IOException {
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <base64>");
      } else {
         String tag = this.sbuf.toString();
         if(tag.equals("null")) {
            this.expectEndTag("null");
            return null;
         } else if(tag.equals("base64")) {
            this.sbuf.setLength(0);
            byte[] value = this.parseBytes();
            this.expectEndTag("base64");
            return value;
         } else {
            throw this.expectBeginTag("base64", tag);
         }
      }
   }

   public Object readObject(Class expectedClass) throws IOException {
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <tag>");
      } else {
         String tag = this.sbuf.toString();
         if(tag.equals("null")) {
            this.expectEndTag("null");
            return null;
         } else {
            int type1;
            if(tag.equals("boolean")) {
               type1 = this.parseInt();
               this.expectEndTag("boolean");
               return new Boolean(type1 != 0);
            } else if(tag.equals("int")) {
               type1 = this.parseInt();
               this.expectEndTag("int");
               return new Integer(type1);
            } else {
               long type2;
               if(tag.equals("long")) {
                  type2 = this.parseLong();
                  this.expectEndTag("long");
                  return new Long(type2);
               } else {
                  String type;
                  if(tag.equals("string")) {
                     this.sbuf.setLength(0);
                     this.parseString(this.sbuf);
                     type = this.sbuf.toString();
                     this.expectEndTag("string");
                     return type;
                  } else if(tag.equals("xml")) {
                     this.sbuf.setLength(0);
                     this.parseString(this.sbuf);
                     type = this.sbuf.toString();
                     this.expectEndTag("xml");
                     return type;
                  } else if(tag.equals("date")) {
                     if(this.utcCalendar == null) {
                        this.utcCalendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
                     }

                     type2 = this.parseDate(this.utcCalendar);
                     this.expectEndTag("date");
                     return new Date(type2);
                  } else if(tag.equals("map")) {
                     type = this.readType();
                     return this.readMap(expectedClass, type);
                  } else if(tag.equals("list")) {
                     type = this.readType();
                     int url1 = this.readLength();
                     return this.readList(expectedClass, type, url1);
                  } else if(tag.equals("ref")) {
                     type1 = this.parseInt();
                     this.expectEndTag("ref");
                     return this.refs.elementAt(type1);
                  } else if(tag.equals("remote")) {
                     type = this.readType();
                     String url = this.readString();
                     this.expectEndTag("remote");
                     return this.resolveRemote(type, url);
                  } else {
                     return this.readExtensionObject(expectedClass, tag);
                  }
               }
            }
         }
      }
   }

   public String readType() throws IOException {
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <type>");
      } else {
         String tag = this.sbuf.toString();
         if(!tag.equals("type")) {
            throw new BurlapProtocolException("expected <type>");
         } else {
            this.sbuf.setLength(0);
            this.parseString(this.sbuf);
            String value = this.sbuf.toString();
            this.expectEndTag("type");
            return value;
         }
      }
   }

   public int readLength() throws IOException {
      this.expectStartTag("length");
      int ch = this.skipWhitespace();
      this.peek = ch;
      if(ch == 60) {
         this.expectEndTag("length");
         return -1;
      } else {
         int value = this.parseInt();
         this.expectEndTag("length");
         return value;
      }
   }

   public Object resolveRemote(String type, String url) throws IOException {
      return new BurlapRemote(type, url);
   }

   public Hashtable readFault() throws IOException {
      this.expectStartTag("fault");
      Hashtable map = new Hashtable();

      while(this.parseTag()) {
         this.peekTag = true;
         Object key = this.readObject((Class)null);
         Object value = this.readObject((Class)null);
         if(key != null && value != null) {
            map.put(key, value);
         }
      }

      if(!this.sbuf.toString().equals("fault")) {
         throw new BurlapProtocolException("expected </fault>");
      } else {
         return map;
      }
   }

   public Object readMap(Class expectedClass, String type) throws IOException {
      Hashtable map = new Hashtable();
      if(this.refs == null) {
         this.refs = new Vector();
      }

      this.refs.addElement(map);

      while(this.parseTag()) {
         this.peekTag = true;
         Object key = this.readObject((Class)null);
         Object value = this.readObject((Class)null);
         map.put(key, value);
      }

      if(!this.sbuf.toString().equals("map")) {
         throw new BurlapProtocolException("expected </map>");
      } else {
         return map;
      }
   }

   protected Object readExtensionObject(Class expectedClass, String tag) throws IOException {
      throw new BurlapProtocolException("unknown object tag <" + tag + ">");
   }

   public Object readList(Class expectedClass, String type, int length) throws IOException {
      Vector list = new Vector();
      if(this.refs == null) {
         this.refs = new Vector();
      }

      this.refs.addElement(list);

      while(this.parseTag()) {
         this.peekTag = true;
         Object value = this.readObject((Class)null);
         list.addElement(value);
      }

      if(!this.sbuf.toString().equals("list")) {
         throw new BurlapProtocolException("expected </list>");
      } else {
         return list;
      }
   }

   protected int parseInt() throws IOException {
      byte sign = 1;
      int value = 0;
      int ch = this.skipWhitespace();
      if(ch == 43) {
         ch = this.read();
      } else if(ch == 45) {
         sign = -1;
         ch = this.read();
      }

      while(ch >= 48 && ch <= 57) {
         value = 10 * value + ch - 48;
         ch = this.read();
      }

      this.peek = ch;
      return sign * value;
   }

   protected long parseLong() throws IOException {
      long sign = 1L;
      long value = 0L;
      int ch = this.skipWhitespace();
      if(ch == 43) {
         ch = this.read();
      } else if(ch == 45) {
         sign = -1L;
         ch = this.read();
      }

      while(ch >= 48 && ch <= 57) {
         value = 10L * value + (long)ch - 48L;
         ch = this.read();
      }

      this.peek = ch;
      return sign * value;
   }

   protected long parseDate(Calendar calendar) throws IOException {
      int ch = this.skipWhitespace();
      int year = 0;

      int month;
      for(month = 0; month < 4; ++month) {
         if(ch < 48 || ch > 57) {
            throw this.expectedChar("year", ch);
         }

         year = 10 * year + ch - 48;
         ch = this.read();
      }

      month = 0;

      int day;
      for(day = 0; day < 2; ++day) {
         if(ch < 48 || ch > 57) {
            throw this.expectedChar("month", ch);
         }

         month = 10 * month + ch - 48;
         ch = this.read();
      }

      day = 0;

      int hour;
      for(hour = 0; hour < 2; ++hour) {
         if(ch < 48 || ch > 57) {
            throw this.expectedChar("day", ch);
         }

         day = 10 * day + ch - 48;
         ch = this.read();
      }

      if(ch != 84) {
         throw this.expectedChar("`T\'", ch);
      } else {
         ch = this.read();
         hour = 0;

         int minute;
         for(minute = 0; minute < 2; ++minute) {
            if(ch < 48 || ch > 57) {
               throw this.expectedChar("hour", ch);
            }

            hour = 10 * hour + ch - 48;
            ch = this.read();
         }

         minute = 0;

         int second;
         for(second = 0; second < 2; ++second) {
            if(ch < 48 || ch > 57) {
               throw this.expectedChar("minute", ch);
            }

            minute = 10 * minute + ch - 48;
            ch = this.read();
         }

         second = 0;

         for(int i = 0; i < 2; ++i) {
            if(ch < 48 || ch > 57) {
               throw this.expectedChar("second", ch);
            }

            second = 10 * second + ch - 48;
            ch = this.read();
         }

         while(ch > 0 && ch != 60) {
            ch = this.read();
         }

         this.peek = ch;
         calendar.set(1, year);
         calendar.set(2, month - 1);
         calendar.set(5, day);
         calendar.set(11, hour);
         calendar.set(12, minute);
         calendar.set(13, second);
         calendar.set(14, 0);
         return calendar.getTime().getTime();
      }
   }

   protected String parseString() throws IOException {
      StringBuffer sbuf = new StringBuffer();
      return this.parseString(sbuf).toString();
   }

   protected StringBuffer parseString(StringBuffer sbuf) throws IOException {
      int ch;
      for(ch = this.read(); ch >= 0 && ch != 60; ch = this.read()) {
         int ch1;
         if(ch == 38) {
            ch = this.read();
            if(ch == 35) {
               ch = this.read();
               if(ch >= 48 && ch <= 57) {
                  for(ch1 = 0; ch >= 48 && ch <= 57; ch = this.read()) {
                     ch1 = 10 * ch1 + ch - 48;
                  }

                  sbuf.append((char)ch1);
               }
            } else {
               StringBuffer ch11;
               for(ch11 = new StringBuffer(); ch >= 97 && ch <= 122; ch = this.read()) {
                  ch11.append((char)ch);
               }

               String ch2 = ch11.toString();
               if(ch2.equals("amp")) {
                  sbuf.append('&');
               } else if(ch2.equals("apos")) {
                  sbuf.append('\'');
               } else if(ch2.equals("quot")) {
                  sbuf.append('\"');
               } else if(ch2.equals("lt")) {
                  sbuf.append('<');
               } else {
                  if(!ch2.equals("gt")) {
                     throw new BurlapProtocolException("unknown XML entity &" + ch2 + "; at `" + (char)ch + "\'");
                  }

                  sbuf.append('>');
               }
            }

            if(ch != 59) {
               throw this.expectedChar("\';\'", ch);
            }
         } else if(ch < 128) {
            sbuf.append((char)ch);
         } else {
            int ch21;
            if((ch & 224) == 192) {
               ch1 = this.read();
               ch21 = ((ch & 31) << 6) + (ch1 & 63);
               sbuf.append((char)ch21);
            } else {
               if((ch & 240) != 224) {
                  throw new BurlapProtocolException("bad utf-8 encoding");
               }

               ch1 = this.read();
               ch21 = this.read();
               int v = ((ch & 15) << 12) + ((ch1 & 63) << 6) + (ch21 & 63);
               sbuf.append((char)v);
            }
         }
      }

      this.peek = ch;
      return sbuf;
   }

   protected byte[] parseBytes() throws IOException {
      ByteArrayOutputStream bos = new ByteArrayOutputStream();
      this.parseBytes(bos);
      return bos.toByteArray();
   }

   protected ByteArrayOutputStream parseBytes(ByteArrayOutputStream bos) throws IOException {
      int ch;
      for(ch = this.read(); ch >= 0 && ch != 60; ch = this.read()) {
         int b2 = this.read();
         int b3 = this.read();
         int b4 = this.read();
         int chunk;
         if(b4 != 61) {
            chunk = (base64Decode[ch] << 18) + (base64Decode[b2] << 12) + (base64Decode[b3] << 6) + base64Decode[b4];
            bos.write(chunk >> 16);
            bos.write(chunk >> 8);
            bos.write(chunk);
         } else if(b3 != 61) {
            chunk = (base64Decode[ch] << 12) + (base64Decode[b2] << 6) + base64Decode[b3];
            bos.write(chunk >> 8);
            bos.write(chunk);
         } else {
            chunk = (base64Decode[ch] << 6) + base64Decode[b2];
            bos.write(chunk);
         }
      }

      if(ch == 60) {
         this.peek = ch;
      }

      return bos;
   }

   protected void expectStartTag(String tag) throws IOException {
      if(!this.parseTag()) {
         throw new BurlapProtocolException("expected <" + tag + ">");
      } else if(!this.sbuf.toString().equals(tag)) {
         throw new BurlapProtocolException("expected <" + tag + "> at <" + this.sbuf + ">");
      }
   }

   protected void expectEndTag(String tag) throws IOException {
      if(this.parseTag()) {
         throw new BurlapProtocolException("expected </" + tag + ">");
      } else if(!this.sbuf.toString().equals(tag)) {
         throw new BurlapProtocolException("expected </" + tag + "> at </" + this.sbuf + ">");
      }
   }

   protected boolean parseTag() throws IOException {
      if(this.peekTag) {
         this.peekTag = false;
         return true;
      } else {
         int ch = this.skipWhitespace();
         boolean isStartTag = true;
         if(ch != 60) {
            throw this.expectedChar("\'<\'", ch);
         } else {
            ch = this.read();
            if(ch == 47) {
               isStartTag = false;
               ch = this.is.read();
            }

            if(!this.isTagChar(ch)) {
               throw this.expectedChar("tag", ch);
            } else {
               this.sbuf.setLength(0);

               while(this.isTagChar(ch)) {
                  this.sbuf.append((char)ch);
                  ch = this.read();
               }

               if(ch != 62) {
                  throw this.expectedChar("\'>\'", ch);
               } else {
                  return isStartTag;
               }
            }
         }
      }
   }

   protected IOException expectedChar(String expect, int actualChar) {
      return new BurlapProtocolException("expected " + expect + " at " + (char)actualChar + "\'");
   }

   protected IOException expectBeginTag(String expect, String tag) {
      return new BurlapProtocolException("expected <" + expect + "> at <" + tag + ">");
   }

   private boolean isTagChar(int ch) {
      return ch >= 97 && ch <= 122 || ch >= 65 && ch <= 90 || ch >= 48 && ch <= 57 || ch == 58 || ch == 45;
   }

   protected int skipWhitespace() throws IOException {
      int ch;
      for(ch = this.read(); ch == 32 || ch == 9 || ch == 10 || ch == 13; ch = this.read()) {
         ;
      }

      return ch;
   }

   protected boolean isWhitespace(int ch) throws IOException {
      return ch == 32 || ch == 9 || ch == 10 || ch == 13;
   }

   protected int read() throws IOException {
      if(this.peek > 0) {
         int value = this.peek;
         this.peek = 0;
         return value;
      } else {
         return this.is.read();
      }
   }

   static {
      int i;
      for(i = 65; i <= 90; ++i) {
         base64Decode[i] = i - 65;
      }

      for(i = 97; i <= 122; ++i) {
         base64Decode[i] = i - 97 + 26;
      }

      for(i = 48; i <= 57; ++i) {
         base64Decode[i] = i - 48 + 52;
      }

      base64Decode[43] = 62;
      base64Decode[47] = 63;
   }
}
