package com.wb.tool;

import com.wb.common.Var;
import com.wb.util.FileUtil;
import com.wb.util.JsonUtil;
import com.wb.util.StringUtil;
import com.wb.util.SysUtil;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.json.JSONArray;
import org.json.JSONObject;

public class MgDataOutput {

   public static void outputExc(JSONArray headers, JSONArray records, String title, boolean neptune, boolean isAjax, String finalXlsxPath, String filename) throws Exception {
      HSSFWorkbook book = null;
      FileOutputStream out = null;
      byte startRow = 0;
      String sheetName = "VolckerReport";

      try {
         book = new HSSFWorkbook();
         Sheet sheet = book.createSheet();
         if(!StringUtil.isEmpty(sheetName)) {
            book.setSheetName(book.getSheetIndex(sheet), sheetName);
         }

         if(title != null) {
            startRow = 1;
         }

         Object[] values = createHeaders(sheet, headers, startRow, neptune);
         int headerCols = ((Integer)values[0]).intValue();
         int headerRows = ((Integer)values[1]).intValue();
         JSONArray fields = (JSONArray)values[2];
         if(title != null) {
            createTitle(sheet, title, headerCols);
         }

         int startRow1 = startRow + headerRows;
         if(Var.getBool("sys.service.excel.freezePane")) {
            sheet.createFreezePane(0, startRow1);
         }

         createRecord(sheet, records, fields, startRow1, filename);
         out = new FileOutputStream(finalXlsxPath);
         book.write(out);
      } catch (Exception var20) {
         var20.printStackTrace();
      } finally {
         book.close();
         if(out != null) {
            out.close();
         }

      }

   }

   public static StringBuilder outputDat(JSONArray headers, JSONArray records, String filename) throws Exception {
      StringBuilder text = new StringBuilder();
      JSONArray leafs = new JSONArray();
      String lineSeparator = "\n";
      leafs.put(0);
      markParents(leafs, headers, (JSONObject)null, 0);
      leafs.remove(0);
      int j = leafs.length();

      for(int i = 0; i < j; ++i) {
         if(i > 0) {
            text.append('|');
         }

         text.append(leafs.getJSONObject(i).optString("text"));
      }

      text.append(lineSeparator);
      getDatContent(text, records, leafs, false, filename, lineSeparator);
      return text;
   }

   private static void createRecord(Sheet sheet, JSONArray records, JSONArray fields, int startRow, String filename) {
      int j = records.length();
      int l = fields.length();
      String[] fieldNames = new String[l];
      Workbook book = sheet.getWorkbook();
      Object[] cellStyles = createCellStyle(book, "text");
      CellStyle baseStyle = (CellStyle)cellStyles[0];
      CellStyle[] colStyles = new CellStyle[l];
      short rowHeight = ((Short)cellStyles[1]).shortValue();

      int k;
      for(k = 0; k < l; ++k) {
         JSONObject field = fields.getJSONObject(k);
         fieldNames[k] = field.optString("field");
         CellStyle style = book.createCellStyle();
         style.cloneStyleFrom(baseStyle);
         style.setAlignment(getAlignment(field.optString("align"), (short)1));
         if(Boolean.TRUE.equals(field.opt("autoWrap"))) {
            style.setWrapText(true);
         }

         colStyles[k] = style;
      }

      int i;
      Cell cell;
      Row row;
      for(i = 0; i < j; ++i) {
         row = sheet.createRow(startRow + i);
         row.setHeight(rowHeight);
         JSONObject record = (JSONObject)records.opt(i);

         for(k = 0; k < l; ++k) {
            cell = row.createCell(k);
            cell.setCellStyle(colStyles[k]);
            Object value = JsonUtil.opt(record, fieldNames[k]);
            if(value != null) {
               cell.setCellValue(value.toString());
            }
         }
      }

      row = sheet.createRow(startRow + j);
      row.setHeight(rowHeight);

      for(i = 0; i < l; ++i) {
         cell = row.createCell(i);
         cell.setCellStyle(colStyles[i]);
         if(i == 0) {
            cell.setCellValue("#EOF#");
         }

         if(i == 1) {
            cell.setCellValue("VV_2162966_" + filename + "_1_FED_MSHX.dat");
         }

         if(i == 2) {
            cell.setCellValue((double)(startRow + j + 1));
         }

         if(i == 3) {
            cell.setCellValue(filename);
         }

         if(i == 4) {
            cell.setCellValue("volcker@morganstanleyhuaxin.com");
         }

         if(i == 5) {
            SimpleDateFormat myFmt = new SimpleDateFormat("yyyyMMdd:HH:mm");
            cell.setCellValue(myFmt.format(new Date()));
         }
      }

   }

   private static void createTitle(Sheet sheet, String title, int headerCols) {
      Row row = sheet.createRow(0);
      Object[] styles = createCellStyle(sheet.getWorkbook(), "title");
      row.setHeight(((Short)styles[1]).shortValue());
      sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, headerCols - 1));
      Cell cell = row.createCell(0);
      cell.setCellStyle((CellStyle)styles[0]);
      cell.setCellValue(title);
   }

   private static Object[] createHeaders(Sheet sheet, JSONArray headers, int startRow, boolean neptune) {
      Workbook book = sheet.getWorkbook();
      JSONArray processedHeaders = new JSONArray();
      Object[] values = prepareHeaders(sheet, headers, processedHeaders, startRow, neptune);
      Cell[][] cells = (Cell[][])values[0];
      Object[] styles = createCellStyle(book, "header");
      CellStyle baseStyle = (CellStyle)styles[0];
      int j = processedHeaders.length();

      for(int i = 0; i < j; ++i) {
         JSONObject header = processedHeaders.getJSONObject(i);
         int x = header.getInt("x");
         int y = header.getInt("y");
         int colspan = Math.max(header.getInt("colspan"), 0);
         int rowspan = Math.max(header.getInt("rowspan"), 0);
         if(colspan > 0 || rowspan > 0) {
            sheet.addMergedRegion(new CellRangeAddress(y + startRow, y + startRow + rowspan, x, x + colspan));
         }

         Cell cell = cells[x][y];
         CellStyle style = book.createCellStyle();
         style.cloneStyleFrom(baseStyle);
         style.setAlignment(getAlignment(header.optString("titleAlign"), (short)(header.has("child")?2:1)));
         cell.setCellStyle(style);
         cell.setCellValue(header.optString("text"));
      }

      Object[] result = new Object[]{Integer.valueOf(cells.length), Integer.valueOf(cells[0].length), values[1]};
      return result;
   }

   private static Object[] prepareHeaders(Sheet sheet, JSONArray rawHeaders, JSONArray processedHeaders, int startRow, boolean neptune) {
      JSONArray leafs = new JSONArray();
      Object[] result = new Object[2];
      int flexWidth = Var.getInt("sys.service.excel.flexColumnMaxWidth");
      Object[] styles = createCellStyle(sheet.getWorkbook(), "header");
      CellStyle style = (CellStyle)styles[0];
      short rowHeight = ((Short)styles[1]).shortValue();
      double rate;
      if(neptune) {
         rate = 32.06D;
      } else {
         rate = 36.55D;
      }

      leafs.put(0);
      markParents(leafs, rawHeaders, (JSONObject)null, 0);
      int maxDepth = leafs.getInt(0);
      leafs.remove(0);
      int j = leafs.length();

      for(int i = 0; i < j; ++i) {
         JSONObject node = leafs.getJSONObject(i);
         int width;
         if(node.has("width")) {
            width = node.getInt("width");
         } else if(node.has("flex")) {
            width = flexWidth;
         } else {
            width = 100;
         }

         sheet.setColumnWidth(i, (int)((double)width * rate));
         node.put("rowspan", maxDepth - node.getInt("y"));

         do {
            node.put("colspan", node.getInt("colspan") + 1);
            if(!node.has("x")) {
               node.put("x", i);
               processedHeaders.put(node);
            }
         } while((node = (JSONObject)node.opt("parent")) != null);
      }

      ++maxDepth;
      Cell[][] cells = new Cell[j][maxDepth];

      for(int k = 0; k < maxDepth; ++k) {
         Row row = sheet.createRow(k + startRow);
         row.setHeight(rowHeight);

         for(int l = 0; l < j; ++l) {
            Cell cell = row.createCell(l);
            cell.setCellStyle(style);
            cells[l][k] = cell;
         }
      }

      result[0] = cells;
      result[1] = leafs;
      return result;
   }

   private static void markParents(JSONArray leafs, JSONArray headers, JSONObject parent, int depth) {
      int j = headers.length();
      leafs.put(0, Math.max(leafs.getInt(0), depth));

      for(int i = 0; i < j; ++i) {
         JSONObject header = headers.getJSONObject(i);
         header.put("y", depth);
         header.put("colspan", -1);
         header.put("rowspan", -1);
         if(parent != null) {
            header.put("parent", parent);
            parent.put("child", header);
         }

         JSONArray items = (JSONArray)header.opt("items");
         if(items != null) {
            markParents(leafs, items, header, depth + 1);
         } else {
            leafs.put(header);
         }
      }

   }

   public static Workbook getBook(String string) throws Exception {
      ByteArrayInputStream is = null;
      if(!StringUtil.isEmpty(string)) {
         is = new ByteArrayInputStream(StringUtil.decodeBase64(string));
      }

      return (Workbook)(Var.getBool("sys.service.excel.xlsx")?(is == null?new XSSFWorkbook():new XSSFWorkbook(is)):(is == null?new HSSFWorkbook():new HSSFWorkbook(is)));
   }

   public static Object[] createCellStyle(Workbook book, String type) {
      CellStyle style = book.createCellStyle();
      Font font = book.createFont();
      String fontName = Var.getString("sys.service.excel." + type + ".fontName");
      int fontHeight = Var.getInt("sys.service.excel." + type + ".fontHeight");
      double rowHeight = Var.getDouble("sys.service.excel." + type + ".rowHeight");
      Object[] result = new Object[2];
      if(!fontName.isEmpty()) {
         font.setFontName(fontName);
      }

      font.setBoldweight((short)Var.getInt("sys.service.excel." + type + ".fontWeight"));
      font.setFontHeight((short)fontHeight);
      if(rowHeight < 10.0D) {
         rowHeight *= (double)fontHeight;
      }

      if(!"text".equals(type) && Var.getBool("sys.service.excel." + type + ".wrapText")) {
         style.setWrapText(true);
      }

      String backColor;
      Object[][] colors;
      if("title".equals(type)) {
         backColor = Var.getString("sys.service.excel." + type + ".align");
         if(!backColor.isEmpty()) {
            colors = new Object[][]{{"居中", Short.valueOf((short)2)}, {"左", Short.valueOf((short)1)}, {"右", Short.valueOf((short)3)}, {"居中选择", Short.valueOf((short)6)}, {"填充", Short.valueOf((short)4)}, {"常规", Short.valueOf((short)0)}, {"两端对齐", Short.valueOf((short)5)}};
            style.setAlignment(((Short)SysUtil.getValue(colors, backColor)).shortValue());
         }
      } else if(Var.getBool("sys.service.excel.border")) {
         style.setBorderTop((short)1);
         style.setBorderBottom((short)1);
         style.setBorderLeft((short)1);
         style.setBorderRight((short)1);
      }

      if("header".equals(type)) {
         backColor = Var.getString("sys.service.excel.header.backColor");
         if(!"默认".equals(backColor)) {
            colors = new Object[][]{{"默认", Integer.valueOf(-1)}, {"金色", Short.valueOf((short)51)}, {"灰色", Short.valueOf((short)22)}, {"浅黄", Short.valueOf((short)43)}};
            style.setFillForegroundColor(((Short)SysUtil.getValue(colors, backColor)).shortValue());
            style.setFillPattern((short)1);
         }
      }

      style.setVerticalAlignment((short)1);
      style.setFont(font);
      result[0] = style;
      result[1] = Short.valueOf(Double.valueOf(rowHeight).shortValue());
      return result;
   }

   public static short getAlignment(String align, short defaultAlign) {
      return "right".equals(align)?3:("center".equals(align)?2:("left".equals(align)?1:defaultAlign));
   }

   public static String excelToJson(InputStream inputStream, boolean xlsxFormat) throws Exception {
      int rowIndex = 0;
      ArrayList fieldList = new ArrayList();
      StringBuilder text = new StringBuilder("");
      Object book;
      if(xlsxFormat) {
         book = new XSSFWorkbook(inputStream);
      } else {
         book = new HSSFWorkbook(inputStream);
      }

      try {
         Sheet sheet = ((Workbook)book).getSheetAt(0);

         for(Iterator rows = sheet.rowIterator(); rows.hasNext(); ++rowIndex) {
            if(rowIndex > 1) {
               text.append("\n{");
            } else if(rowIndex > 0) {
               text.append('{');
            }

            Row row = (Row)rows.next();
            Iterator cells = row.cellIterator();

            for(int colIndex = 0; cells.hasNext(); ++colIndex) {
               Cell cell = (Cell)cells.next();
               Object value = getCellValue(cell);
               if(rowIndex == 0) {
                  if(value == null) {
                     throw new NullPointerException("Field name has null value.");
                  }

                  String valueStr = value.toString();
                  int pos = valueStr.indexOf(40);
                  int pos1 = valueStr.indexOf("（");
                  if(pos1 != -1 && (pos == -1 || pos1 < pos)) {
                     pos = pos1;
                  }

                  if(pos == -1) {
                     fieldList.add(valueStr);
                  } else {
                     fieldList.add(valueStr.substring(0, pos));
                  }
               } else {
                  if(colIndex > 0) {
                     text.append(',');
                  }

                  if(colIndex >= fieldList.size()) {
                     throw new RuntimeException("Row " + (rowIndex + 1) + " column " + (colIndex + 1) + " is out of bounds.");
                  }

                  text.append(StringUtil.quote((String)fieldList.get(colIndex)));
                  text.append(':');
                  text.append(StringUtil.encode(value));
               }
            }

            if(rowIndex > 0) {
               text.append('}');
            }
         }
      } finally {
         ((Workbook)book).close();
      }

      return text.toString();
   }

   public static Object getCellValue(Cell cell) {
      switch(cell.getCellType()) {
      case 0:
      case 2:
         if(DateUtil.isCellDateFormatted(cell)) {
            return cell.getDateCellValue();
         }

         return Double.valueOf(cell.getNumericCellValue());
      case 1:
         return cell.getStringCellValue();
      case 3:
      default:
         return null;
      case 4:
         return Boolean.valueOf(cell.getBooleanCellValue());
      }
   }

   public static String toJavaDateFormat(String format, boolean returnDefault) {
      String[] unSupportFormats = new String[]{"N", "S", "D", "w", "z", "W", "t", "L", "o", "O", "P", "T", "Z", "c", "U", "F", "MS", "l", "M", "time", "timestamp"};
      String[][] supportFormats = new String[][]{{"y", "yy"}, {"Y", "yyyy"}, {"m", "MM"}, {"n", "M"}, {"d", "dd"}, {"j", "d"}, {"H", "HH"}, {"h", "hh"}, {"G", "H"}, {"g", "h"}, {"i", "mm"}, {"s", "ss"}, {"u", "SSS"}, {"a", "\'_x\'"}, {"A", "\'_X\'"}};
      String[] var7 = unSupportFormats;
      int var6 = unSupportFormats.length;

      int var5;
      for(var5 = 0; var5 < var6; ++var5) {
         String s = var7[var5];
         if(format.indexOf(s) != -1) {
            return returnDefault?"yyyy-MM-dd":null;
         }
      }

      String[][] var9 = supportFormats;
      var6 = supportFormats.length;

      for(var5 = 0; var5 < var6; ++var5) {
         String[] var8 = var9[var5];
         format = StringUtil.replaceAll(format, var8[0], var8[1]);
      }

      return format;
   }

   public static String toExcelDateFormat(String format, boolean returnDefault) {
      String[] unSupportFormats = new String[]{"N", "S", "w", "z", "W", "t", "L", "o", "u", "O", "P", "T", "Z", "c", "U", "MS", "time", "timestamp"};
      String[][] supportFormats = new String[][]{{"d", "dd"}, {"D", "aaa"}, {"j", "d"}, {"l", "aaaa"}, {"F", "mmmm"}, {"m", "mm"}, {"M", "mmm"}, {"n", "m"}, {"Y", "yyyy"}, {"y", "yy"}, {"a", "am/pm"}, {"A", "AM/PM"}, {"g", "h"}, {"G", "hh"}, {"h", "h"}, {"H", "hh"}, {"i", "mm"}, {"s", "ss"}};
      String[] var7 = unSupportFormats;
      int var6 = unSupportFormats.length;

      int var5;
      for(var5 = 0; var5 < var6; ++var5) {
         String s = var7[var5];
         if(format.indexOf(s) != -1) {
            return returnDefault?"yyyy-mm-dd":null;
         }
      }

      String[][] var9 = supportFormats;
      var6 = supportFormats.length;

      for(var5 = 0; var5 < var6; ++var5) {
         String[] var8 = var9[var5];
         format = StringUtil.replaceAll(format, var8[0], var8[1]);
      }

      return format;
   }

   public static void importToExcel(File file, JSONObject data, OutputStream outputStream) throws Exception {
      boolean isXlsx = FileUtil.getFileExt(file.getName()).toLowerCase().equals("xlsx");
      FileInputStream is = new FileInputStream(file);
      Object book = null;

      try {
         if(isXlsx) {
            book = new XSSFWorkbook(is);
         } else {
            book = new HSSFWorkbook(is);
         }

         Sheet sheet = ((Workbook)book).getSheetAt(0);
         importToSheet(sheet, data);
         ((Workbook)book).write(outputStream);
         ((Workbook)book).close();
      } finally {
         IOUtils.closeQuietly((Closeable)book);
         is.close();
      }

   }

   public static void importToSheet(Sheet sheet, JSONObject data) {
      ArrayList fields = null;
      Iterator rows = sheet.rowIterator();
      int rowIndex = 0;
      boolean strIndex = false;

      for(String field = null; rows.hasNext(); ++rowIndex) {
         Row row = (Row)rows.next();
         Iterator cells = row.cellIterator();
         fields = new ArrayList();

         while(cells.hasNext()) {
            Cell cell = (Cell)cells.next();
            Object value = getCellValue(cell);
            if(value != null) {
               String string = value.toString();
               if(StringUtils.contains(string, "{#") && StringUtils.contains(string, "#}")) {
                  int var12 = StringUtils.indexOf(string, "{#");
                  if(var12 != 0) {
                     field = StringUtils.substringBetween(string, "{#", "#}");
                     value = data.opt(field);
                     if(value != null) {
                        cell.setCellValue(StringUtils.substringBefore(string, "{#") + value.toString() + StringUtils.substringAfter(string, "#}"));
                     } else {
                        cell.setCellValue(StringUtils.substringBefore(string, "{#") + StringUtils.substringAfter(string, "#}"));
                     }
                  } else {
                     string = string.substring(2, string.length() - 2);
                     value = data.opt(string);
                     if(value == JSONObject.NULL) {
                        cell.setCellValue(" ");
                     } else if(value instanceof Number) {
                        cell.setCellValue(((Number)value).doubleValue());
                     } else if(value instanceof Date) {
                        cell.setCellValue((Date)value);
                     } else if(value instanceof Boolean) {
                        cell.setCellValue(((Boolean)value).booleanValue());
                     } else {
                        cell.setCellValue(ObjectUtils.toString(value, ""));
                     }
                  }
               }

               if(StringUtils.contains(string, "{R#") && StringUtils.contains(string, "#}")) {
                  string = string.substring(StringUtils.indexOf(string, "{R#") + 3, string.length() - 2);
                  fields.add(string);
               }
            }
         }

         if(fields.size() > 0) {
            createRows(sheet, data.getJSONArray("rows"), fields, rowIndex);
         }
      }

   }

   private static void createRows(Sheet sheet, JSONArray records, List fields, int startRow) {
      int j = records.length();
      int l = fields.size();

      for(int i = 0; i < j; ++i) {
         Row row = sheet.createRow(startRow + i);
         JSONObject record = (JSONObject)records.opt(i);

         for(int k = 0; k < l; ++k) {
            Cell cell = row.createCell(k);
            Object value = JsonUtil.opt(record, (String)fields.get(k));
            if(value == JSONObject.NULL) {
               cell.setCellValue("");
            }

            if(value instanceof Number) {
               cell.setCellValue(((Number)value).doubleValue());
            } else if(value instanceof Date) {
               cell.setCellValue((Date)value);
            } else if(value instanceof Boolean) {
               cell.setCellValue(((Boolean)value).booleanValue());
            } else {
               cell.setCellValue(ObjectUtils.toString(value, ""));
            }
         }
      }

   }

   private static void getDatContent(StringBuilder buf, JSONArray records, JSONArray fields, boolean hasRowNumber, String filename, String lineSeparator) {
      int j = records.length();
      int m = j + 2;
      int l = fields.length();
      String[] fieldNames = new String[l];
      SimpleDateFormat myFmt = new SimpleDateFormat("yyyyMMdd:HH:mm");
      String lastText = "#EOF#|VV_2162966_" + filename + "_1_FED_MSHX.dat|" + m + "|" + filename + "|volcker@morganstanleyhuaxin.com|" + myFmt.format(new Date());
      String[] aligns = new String[l];
      boolean[] wraps = new boolean[l];

      int k;
      for(k = 0; k < l; ++k) {
         JSONObject field = fields.getJSONObject(k);
         fieldNames[k] = field.optString("field");
         aligns[k] = field.optString("align");
         wraps[k] = Boolean.TRUE.equals(field.opt("autoWrap"));
      }

      for(int i = 0; i < j; ++i) {
         JSONObject record = (JSONObject)records.opt(i);
         if(i > 0) {
            buf.append(lineSeparator);
         }

         for(k = 0; k < l; ++k) {
            Object value = JsonUtil.opt(record, fieldNames[k]);
            if(value == null) {
               buf.append("\t");
            } else {
               if(!value.equals("") && k > 0) {
                  buf.append("|");
               }

               String valueText = value.toString();
               buf.append(valueText);
            }
         }
      }

      buf.append("\n" + lastText);
   }
}
