package com.rw.model.excel;

import com.rw.model.excel.XCell;
import com.rw.model.excel.XCellStyle;
import com.rw.model.excel.XRow;
import com.rw.model.excel.XSheet;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.util.CollectionUtils;

public class ExcelTranslate {

   private static final Logger logger = Logger.getLogger(ExcelTranslate.class);
   private Workbook workbook;
   private FormulaEvaluator evaluator;
   private int maxRow;
   private int maxCol = 0;
   private List sheetList = new ArrayList();
   private List cellRangeAddressList = new ArrayList();


   public List getSheetList() {
      return this.sheetList;
   }

   public void setSheetList(List sheetList) {
      this.sheetList = sheetList;
   }

   private void readMaxCell(Sheet sheet) {
      this.maxCol = 0;
      Iterator rit = sheet.rowIterator();

      while(rit.hasNext()) {
         Row row = (Row)rit.next();
         if(row.getLastCellNum() > this.maxCol) {
            this.maxCol = row.getLastCellNum();
         }
      }

   }

   public Workbook create(String fileName, InputStream fis) throws IOException, InvalidFormatException {
      if(StringUtils.endsWithIgnoreCase(fileName, "xls")) {
         return new HSSFWorkbook(fis);
      } else if(StringUtils.endsWithIgnoreCase(fileName, "xlsx")) {
         return new XSSFWorkbook(fis);
      } else {
         throw new IllegalArgumentException("你的excel版本目前poi解析不了");
      }
   }

   public void readExcel(String fileName, InputStream fis, int maxRow, List sheetNameList) throws Exception {
      try {
         logger.info("---------Excel解析开始--------------");
         this.workbook = WorkbookFactory.create(fis);
         this.evaluator = this.workbook.getCreationHelper().createFormulaEvaluator();
         int e = this.workbook.getNumberOfSheets();
         String sheetName = null;
         Sheet sheet = null;
         XSheet xSheet = null;

         for(int n = 0; n < e; ++n) {
            if(!this.workbook.isSheetHidden(n) && !this.workbook.isSheetVeryHidden(n)) {
               sheet = this.workbook.getSheetAt(n);
               xSheet = new XSheet();
               sheetName = sheet.getSheetName();
               if(sheetNameList != null && !CollectionUtils.isEmpty(sheetNameList) && sheetNameList.contains(sheetName)) {
                  this.readSheet(maxRow, sheet, xSheet);
               } else if(sheetNameList == null || CollectionUtils.isEmpty(sheetNameList)) {
                  this.readSheet(maxRow, sheet, xSheet);
               }
            }
         }

      } catch (FileNotFoundException var10) {
         var10.printStackTrace();
         logger.error(var10.getMessage());
         throw var10;
      } catch (IOException var11) {
         var11.printStackTrace();
         logger.error(var11.getMessage());
         throw var11;
      } catch (InvalidFormatException var12) {
         logger.error(var12.getMessage());
         var12.printStackTrace();
         throw var12;
      }
   }

   private void readSheet(int maxRow, Sheet sheet, XSheet xSheet) {
      xSheet.setLabel(sheet.getSheetName());
      logger.info("--------sheetName=" + sheet.getSheetName() + "-----------");
      xSheet.setDefaultCellWidth(sheet.getDefaultColumnWidth());
      xSheet.setDefaultRowHeight(sheet.getDefaultRowHeight());
      int sheetMergeCount = sheet.getNumMergedRegions();

      int maxRowNum;
      for(maxRowNum = 0; maxRowNum < sheetMergeCount; ++maxRowNum) {
         CellRangeAddress r = sheet.getMergedRegion(maxRowNum);
         this.cellRangeAddressList.add(r);
      }

      this.readMaxCell(sheet);
      xSheet.setMaxColNum(this.maxCol);
      boolean var11 = false;
      if(maxRow != 0) {
         maxRowNum = sheet.getLastRowNum() > maxRow?maxRow:sheet.getLastRowNum();
      } else {
         maxRowNum = sheet.getLastRowNum();
      }

      logger.info("导入Excel最大条数:" + maxRowNum);

      for(int var12 = 0; var12 <= maxRowNum; ++var12) {
         Row row = sheet.getRow(var12);
         if(!row.getZeroHeight()) {
            XRow xRow = new XRow();
            xRow.setRowIndex(var12);

            for(int c = 0; c < this.maxCol; ++c) {
               XCell xcell = this.getCell(sheet, var12, c);
               xRow.getCellList().add(xcell);
            }

            xSheet.getRowList().add(xRow);
         }
      }

      this.sheetList.add(xSheet);
   }

   public void readExcel1Sheet(String fileName, InputStream fis, int maxRow, String sheetName) {
      try {
         this.workbook = WorkbookFactory.create(fis);
         Sheet e = this.workbook.getSheet(sheetName);
         XSheet xSheet = new XSheet();
         xSheet.setLabel(e.getSheetName());
         xSheet.setDefaultCellWidth(e.getDefaultColumnWidth());
         xSheet.setDefaultRowHeight(e.getDefaultRowHeight());
         int sheetMergeCount = e.getNumMergedRegions();

         int maxRowNum;
         for(maxRowNum = 0; maxRowNum < sheetMergeCount; ++maxRowNum) {
            CellRangeAddress r = e.getMergedRegion(maxRowNum);
            this.cellRangeAddressList.add(r);
         }

         this.readMaxCell(e);
         xSheet.setMaxColNum(this.maxCol);
         boolean var18 = false;
         if(maxRow != 0) {
            maxRowNum = e.getLastRowNum() > maxRow?maxRow:e.getLastRowNum();
         } else {
            maxRowNum = e.getLastRowNum();
         }

         for(int var19 = 0; var19 <= maxRowNum; ++var19) {
            Row row = e.getRow(var19);
            if(!row.getZeroHeight()) {
               XRow xRow = new XRow();
               xRow.setRowIndex(var19);

               for(int c = 0; c < this.maxCol; ++c) {
                  XCell xcell = this.getCell(e, var19, c);
                  xRow.getCellList().add(xcell);
               }

               xSheet.getRowList().add(xRow);
            }
         }

         this.sheetList.add(xSheet);
      } catch (FileNotFoundException var14) {
         var14.printStackTrace();
         logger.error(var14.getMessage());
      } catch (IOException var15) {
         var15.printStackTrace();
         logger.error(var15.getMessage());
      } catch (InvalidFormatException var16) {
         logger.error(var16.getMessage());
         var16.printStackTrace();
      } catch (RuntimeException var17) {
         logger.error(var17.getMessage());
         var17.printStackTrace();
      }

   }

   public XCell getCell(Sheet sheet, int rowIndex, int colIndex) {
      XCell c = new XCell();
      c.setColIndex(colIndex);
      c.setRowIndex(rowIndex);
      c.setWidth(sheet.getDefaultColumnWidth());
      Row row = sheet.getRow(rowIndex);
      if(row != null && !row.getZeroHeight()) {
         Cell cell = row.getCell(colIndex);
         if(cell == null) {
            return c;
         } else {
            if(this.isMergedRegion(sheet, rowIndex, colIndex)) {
               int[] mergeRange = this.getMergedRegionRange(sheet, rowIndex, colIndex);
               if(mergeRange[0] == rowIndex && mergeRange[1] == colIndex) {
                  c.setColSpan(mergeRange[3] - mergeRange[1] + 1);
                  c.setRowSpan(mergeRange[2] - mergeRange[0] + 1);
               } else {
                  c.setColSpan(0);
                  c.setRowSpan(0);
               }
            } else {
               c.setColSpan(1);
               c.setRowSpan(1);
            }

            c.setValue(this.getValue(sheet, cell));
            c.setxCellStyle(new XCellStyle(cell.getCellStyle(), sheet.getWorkbook()));
            return c;
         }
      } else {
         return c;
      }
   }

   private Object getValue(Sheet sheet, Cell cell) {
      Cell newCell = null;
      int cellColIndex = cell.getColumnIndex();
      int cellRowIndex = cell.getRowIndex();
      if(this.isMergedRegion(sheet, cellRowIndex, cellColIndex)) {
         newCell = this.getMergedRegionCell(sheet, cellRowIndex, cellColIndex);
         return this.getCellValue(newCell);
      } else {
         return this.getCellValue(cell);
      }
   }

   private Object getCellValue(Cell cell) {
      Object cellValue = "";
      if(cell != null) {
         switch(cell.getCellType()) {
         case 0:
            String cellString = cell.getCellStyle().getDataFormatString();
            SimpleDateFormat e1;
            Date e2;
            if(StringUtils.isNotBlank(cellString) && cellString.contains("h:mm:ss")) {
               e1 = new SimpleDateFormat("HH:mm:ss");
               e2 = cell.getDateCellValue();
               e1.format(e2);
            }

            if(DateUtil.isInternalDateFormat(cell.getCellStyle().getDataFormat())) {
               e1 = new SimpleDateFormat("yyyyMMdd");
               e2 = cell.getDateCellValue();
               e1.format(e2);
            }

            if(DateUtil.isCellDateFormatted(cell)) {
               e1 = new SimpleDateFormat("yyyyMMdd");
               e2 = cell.getDateCellValue();
               cellValue = e1.format(e2);
            } else {
               cell.setCellType(1);
               cellValue = cell.getStringCellValue();
            }
            break;
         case 1:
            cellValue = cell.getRichStringCellValue() == null?"":cell.getRichStringCellValue().getString();
            break;
         case 2:
            try {
               cellValue = this.readNumericCell(cell);
            } catch (Exception var7) {
               try {
                  cellValue = cell.getRichStringCellValue() == null?"":cell.getRichStringCellValue().getString();
               } catch (Exception var6) {
                  cellValue = "{$ERROR_VALUE}";
               }
            }
         case 3:
            break;
         case 4:
            cellValue = Boolean.toString(cell.getBooleanCellValue());
            break;
         case 5:
            cellValue = "非法字符";
            break;
         default:
            cellValue = "未知类型";
         }
      }

      return cellValue;
   }

   private Object readNumericCell(Cell cell) {
      Object result = null;
      cell = this.evaluator.evaluateInCell(cell);
      double value = cell.getNumericCellValue();
      if((double)((int)value) == value) {
         result = Integer.valueOf((int)value);
      } else {
         result = Double.valueOf(value);
      }

      return result;
   }

   private Cell getValue(Sheet sheet, int rowIndex, int colIndex) {
      Cell cell = null;
      Iterator var6 = this.cellRangeAddressList.iterator();

      while(var6.hasNext()) {
         CellRangeAddress range = (CellRangeAddress)var6.next();
         if(range.isInRange(rowIndex, colIndex)) {
            Row row = sheet.getRow(range.getFirstRow());
            cell = row.getCell(range.getFirstColumn());
            break;
         }
      }

      return cell;
   }

   private boolean isMergedRegion(Sheet sheet, int row, int column) {
      int sheetMergeCount = sheet.getNumMergedRegions();

      for(int i = 0; i < sheetMergeCount; ++i) {
         CellRangeAddress ca = sheet.getMergedRegion(i);
         int firstColumn = ca.getFirstColumn();
         int lastColumn = ca.getLastColumn();
         int firstRow = ca.getFirstRow();
         int lastRow = ca.getLastRow();
         if(row >= firstRow && row <= lastRow && column >= firstColumn && column <= lastColumn) {
            return true;
         }
      }

      return false;
   }

   private Cell getMergedRegionCell(Sheet sheet, int row, int column) {
      int sheetMergeCount = sheet.getNumMergedRegions();

      for(int i = 0; i < sheetMergeCount; ++i) {
         CellRangeAddress ca = sheet.getMergedRegion(i);
         int firstColumn = ca.getFirstColumn();
         int lastColumn = ca.getLastColumn();
         int firstRow = ca.getFirstRow();
         int lastRow = ca.getLastRow();
         if(row >= firstRow && row <= lastRow && column >= firstColumn && column <= lastColumn) {
            Row fRow = sheet.getRow(firstRow);
            Cell fCell = fRow.getCell(firstColumn);
            return fCell;
         }
      }

      return null;
   }

   private int[] getMergedRegionRange(Sheet sheet, int row, int column) {
      int[] rtnVal = new int[4];
      int sheetMergeCount = sheet.getNumMergedRegions();

      for(int i = 0; i < sheetMergeCount; ++i) {
         CellRangeAddress ca = sheet.getMergedRegion(i);
         int firstColumn = ca.getFirstColumn();
         int lastColumn = ca.getLastColumn();
         int firstRow = ca.getFirstRow();
         int lastRow = ca.getLastRow();
         if(row >= firstRow && row <= lastRow && column >= firstColumn && column <= lastColumn) {
            rtnVal[0] = firstRow;
            rtnVal[1] = firstColumn;
            rtnVal[2] = lastRow;
            rtnVal[3] = lastColumn;
            break;
         }
      }

      return rtnVal;
   }
}
