package com.rw.aspect;

import com.google.gson.Gson;
import com.rw.annotation.SystemControllerLog;
import com.rw.annotation.SystemDaoLog;
import com.rw.annotation.SystemServiceLog;
import com.rw.service.LogService;
import com.rw.util.HttpKit;
import com.wb.util.LogUtil;
import com.wb.util.WebUtil;
import java.lang.reflect.Method;
import java.util.Enumeration;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Aspect
@Component
public class SystemLogAspect {

   @Resource
   private LogService logService;
   private static final Logger logger = LoggerFactory.getLogger(SystemLogAspect.class);
   private static String RECORD_TYPE_CONTROLLER = "10";
   private static String RECORD_TYPE_SERVICE = "11";
   private static String RECORD_TYPE_DAO = "12";


   @Pointcut("@annotation(com.rw.annotation.SystemControllerLog)")
   public void controllerAspect() {
   }

   @Pointcut("@annotation(com.rw.annotation.SystemServiceLog)")
   public void serviceAspect() {
   }

   @Pointcut("@annotation(com.rw.annotation.SystemDaoLog)")
   public void daoAspect() {
   }

   @Before("controllerAspect()")
   public void doBefore(JoinPoint joinPoint) {
      HttpServletRequest request = HttpKit.getHttpServletRequest();
      HttpServletResponse response = HttpKit.getHttpServletResponse();

      try {
         logger.info("=====前置通知开始=====");
         logger.info("请求方法:" + joinPoint.getTarget().getClass().getName() + "." + joinPoint.getSignature().getName() + "()");
         logger.info("方法描述:" + getControllerMethodDescription(joinPoint, request));
         logger.info("请求人:" + WebUtil.fetch(request, "sys.username"));
         logger.info("请求IP:" + request.getRemoteAddr());
         LogUtil.info(request, getControllerMethodDescription(joinPoint, request));
         logger.info("=====前置通知结束=====");
      } catch (Exception var5) {
         logger.error("==前置通知异常==");
         logger.error("异常信息:{}", var5.getMessage());
      }

   }

   @AfterThrowing(
      pointcut = "serviceAspect()",
      throwing = "e"
   )
   public void doAfterThrowing(JoinPoint joinPoint, Throwable e) {
      HttpServletRequest request = HttpKit.getHttpServletRequest();
      HttpServletResponse response = HttpKit.getHttpServletResponse();
      Gson gson = new Gson();
      String params = "";
      if(joinPoint.getArgs() != null && joinPoint.getArgs().length > 0) {
         for(int ex = 0; ex < joinPoint.getArgs().length; ++ex) {
            params = params + gson.toJson(joinPoint.getArgs()[ex]) + ";";
         }
      }

      try {
         logger.info("=====异常通知开始=====");
         logger.info("异常代码:" + e.getClass().getName());
         logger.info("异常信息:" + e.getMessage());
         logger.info("异常方法:" + joinPoint.getTarget().getClass().getName() + "." + joinPoint.getSignature().getName() + "()");
         logger.info("方法描述:" + getServiceMthodDescription(joinPoint, e));
         logger.info("请求人:" + WebUtil.fetch(request, "sys.username"));
         logger.info("请求IP:" + request.getRemoteAddr());
         logger.info("请求参数:" + params);
         LogUtil.info(request, getServiceMthodDescription(joinPoint, e));
         logger.info("=====异常通知结束=====");
      } catch (Exception var8) {
         logger.error("==异常通知异常==");
         logger.error("异常信息:{}", var8.getMessage());
      }

      logger.error("异常方法:{}异常代码:{}异常信息:{}参数:{}", new Object[]{joinPoint.getTarget().getClass().getName() + joinPoint.getSignature().getName(), e.getClass().getName(), e.getMessage(), params});
   }

   @Around("daoAspect()")
   public Object around(ProceedingJoinPoint joinPoint) {
      HttpServletRequest request = HttpKit.getHttpServletRequest();
      HttpServletResponse response = HttpKit.getHttpServletResponse();
      Gson gson = new Gson();

      try {
         String e = getDaoMethodDescription(joinPoint);
         LogUtil.info(request, e);
         Object result = joinPoint.proceed();
         String resultMsg = "执行成功！";
         if(result != null) {
            resultMsg = "执行结果:" + gson.toJson(result);
            if(resultMsg.length() > 2000) {
               resultMsg = resultMsg.substring(0, 2000);
            }
         }

         LogUtil.info(request, resultMsg);
         return result;
      } catch (Exception var8) {
         logger.error("==异常通知异常==");
         logger.error("异常信息:{}", var8.getMessage());
      } catch (Throwable var9) {
         logger.error("==异常通知异常==");
         logger.error("异常信息:{}", var9.getMessage());
      }

      return null;
   }

   public static String getServiceMthodDescription(JoinPoint joinPoint, Throwable e) throws Exception {
      String targetName = joinPoint.getTarget().getClass().getName();
      String methodName = joinPoint.getSignature().getName();
      Object[] arguments = joinPoint.getArgs();
      Class targetClass = Class.forName(targetName);
      Method[] methods = targetClass.getMethods();
      String exCode = e.getClass().getName();
      String exMsg = e.getMessage();
      StringBuffer description = new StringBuffer();
      Method[] var13 = methods;
      int var12 = methods.length;

      for(int desc = 0; desc < var12; ++desc) {
         Method gson = var13[desc];
         if(gson.getName().equals(methodName)) {
            Class[] clazzs = gson.getParameterTypes();
            if(clazzs.length == arguments.length) {
               description.append(((SystemServiceLog)gson.getAnnotation(SystemServiceLog.class)).description());
               description.append("发生异常===>异常代码:").append(exCode).append(";异常信息:").append(exMsg);
               break;
            }
         }
      }

      Gson var15 = new Gson();
      description.append(";参数信息:").append(var15.toJson(arguments));
      String var16 = description.toString();
      if(var16.length() > 2000) {
         var16 = description.substring(0, 2000);
      }

      return var16;
   }

   public static String getControllerMethodDescription(JoinPoint joinPoint, HttpServletRequest request) throws Exception {
      String targetName = joinPoint.getTarget().getClass().getName();
      String methodName = joinPoint.getSignature().getName();
      Object[] arguments = joinPoint.getArgs();
      Class targetClass = Class.forName(targetName);
      Method[] methods = targetClass.getMethods();
      StringBuffer description = new StringBuffer();
      Method[] var11 = methods;
      int key = methods.length;

      for(int desc = 0; desc < key; ++desc) {
         Method bf = var11[desc];
         if(bf.getName().equals(methodName)) {
            Class[] clazzs = bf.getParameterTypes();
            if(clazzs.length == arguments.length) {
               description.append(((SystemControllerLog)bf.getAnnotation(SystemControllerLog.class)).description());
               break;
            }
         }
      }

      StringBuffer var13 = new StringBuffer();
      if(arguments != null && arguments.length > 0) {
         Enumeration var14 = request.getParameterNames();

         while(var14.hasMoreElements()) {
            String var16 = (String)var14.nextElement();
            var13.append(var16).append("=");
            var13.append(request.getParameter(var16)).append("&");
         }

         if(StringUtils.isBlank(var13.toString())) {
            var13.append(request.getQueryString());
         }
      }

      if(var13.length() > 0) {
         description.append("===>参数信息:").append(var13);
      }

      String var15 = description.toString();
      if(var15.length() > 2000) {
         var15 = description.substring(0, 2000);
      }

      return var15;
   }

   public static String getDaoMethodDescription(JoinPoint joinPoint) throws Exception {
      String targetName = joinPoint.getTarget().getClass().getName();
      String methodName = joinPoint.getSignature().getName();
      Object[] arguments = joinPoint.getArgs();
      Class targetClass = Class.forName(targetName);
      Method[] methods = targetClass.getMethods();
      StringBuffer description = new StringBuffer();
      Method[] var10 = methods;
      int var9 = methods.length;

      for(int desc = 0; desc < var9; ++desc) {
         Method gson = var10[desc];
         if(gson.getName().equals(methodName)) {
            Class[] clazzs = gson.getParameterTypes();
            if(clazzs.length == arguments.length) {
               description.append(((SystemDaoLog)gson.getAnnotation(SystemDaoLog.class)).description());
               break;
            }
         }
      }

      Gson var12 = new Gson();
      description.append(";参数信息:").append(var12.toJson(arguments));
      String var13 = description.toString();
      if(var13.length() > 2000) {
         var13 = description.substring(0, 2000);
      }

      return var13;
   }
}
