package com.merlin204.tde.client.gui.npc.dialogue;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.function.Predicate;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.player.Player;

public class DialogueTree {

   private final Entity boundEntity;
   private final String entityId;
   private DialogueTree.DialogueNode rootNode;


   public DialogueTree(Entity boundEntity) {
      this.boundEntity = boundEntity;
      ResourceLocation registryName = EntityType.m_20613_(boundEntity.m_6095_());
      this.entityId = registryName.toString().replace(':', '.');
   }

   public void setRootNode(DialogueTree.DialogueNode rootNode) {
      this.rootNode = rootNode;
      this.registerNodeAndChildren(rootNode);
   }

   private void registerNodeAndChildren(DialogueTree.DialogueNode node) {
      node.setTree(this);
      Iterator var2 = node.getChildren().iterator();

      while(var2.hasNext()) {
         DialogueTree.DialogueNode child = (DialogueTree.DialogueNode)var2.next();
         this.registerNodeAndChildren(child);
      }

   }

   public DialogueTree.DialogueNode getRootNode() {
      return this.rootNode;
   }

   public Entity getBoundEntity() {
      return this.boundEntity;
   }

   public abstract static class DialogueNode {

      private final String id;
      private final List children = new ArrayList();
      private final List conditions = new ArrayList();
      public DialogueTree tree;


      public DialogueNode(String id) {
         this.id = id;
      }

      void setTree(DialogueTree tree) {
         this.tree = tree;
      }

      public void addChild(DialogueTree.DialogueNode child) {
         this.children.add(child);
      }

      public void addCondition(Predicate condition) {
         this.conditions.add(condition);
      }

      public List getChildren() {
         return Collections.unmodifiableList(this.children);
      }

      public List getAvailableChildren(Player player) {
         ArrayList available = new ArrayList();
         Iterator var3 = this.children.iterator();

         while(var3.hasNext()) {
            DialogueTree.DialogueNode child = (DialogueTree.DialogueNode)var3.next();
            if(child.isAvailable(player)) {
               available.add(child);
            }
         }

         return available;
      }

      public boolean isAvailable(Player player) {
         Iterator var2 = this.conditions.iterator();

         Predicate condition;
         do {
            if(!var2.hasNext()) {
               return true;
            }

            condition = (Predicate)var2.next();
         } while(condition.test(player));

         return false;
      }

      protected String getLangKey() {
         String var10000 = this.tree.entityId;
         return this.id;
      }

      protected String getLocalizedText() {
         String key = this.getLangKey();
         return key;
      }

      public abstract String getOptionText(Player var1);

      public abstract String getResponseText(Player var1);

      public abstract void onSelected(Player var1);

      public String getId() {
         return this.id;
      }
   }

   public static class DialogueSystem {

      private static final Map entityTrees = new WeakHashMap();


      public static void registerEntityTree(Entity entity, DialogueTree tree) {
         entityTrees.put(entity, tree);
      }

      public static DialogueTree getTreeForEntity(Entity entity) {
         return (DialogueTree)entityTrees.get(entity);
      }

   }

   public static class Option {

      private final String optionText;
      private final String responseText;
      private final DialogueTree.DialogueNode targetNode;


      public Option(String optionText, String responseText, DialogueTree.DialogueNode targetNode) {
         this.optionText = optionText;
         this.responseText = responseText;
         this.targetNode = targetNode;
      }

      public String getOptionText() {
         return this.optionText;
      }

      public String getResponseText() {
         return this.responseText;
      }

      public DialogueTree.DialogueNode getTargetNode() {
         return this.targetNode;
      }
   }

   public static class OptionNode extends DialogueTree.DialogueNode {

      public OptionNode(String id) {
         super(id);
      }

      public String getOptionText(Player player) {
         return this.getLocalizedText();
      }

      public String getResponseText(Player player) {
         return null;
      }

      public void onSelected(Player player) {
      }

      public List getOptions(Player player) {
         ArrayList options = new ArrayList();
         Iterator var3 = this.getAvailableChildren(player).iterator();

         while(var3.hasNext()) {
            DialogueTree.DialogueNode child = (DialogueTree.DialogueNode)var3.next();
            options.add(new DialogueTree.Option(child.getOptionText(player), child.getResponseText(player), child));
         }

         return options;
      }
   }

   public static class TextNode extends DialogueTree.DialogueNode {

      private final String responseKey;


      public TextNode(String id) {
         this(id, id);
      }

      public TextNode(String id, String responseKey) {
         super(id);
         this.responseKey = responseKey;
      }

      public String getOptionText(Player player) {
         return this.getLocalizedText();
      }

      public String getResponseText(Player player) {
         String var10000 = this.tree.entityId;
         String key = this.responseKey;
         return key;
      }

      public void onSelected(Player player) {
      }
   }
}
