package com.logistics.fisher.modbus;

import com.logistics.fisher.modbus.entity.exception.ConnectionException;
import com.logistics.fisher.modbus.handler.ModbusChannelInitializer;
import com.logistics.fisher.modbus.handler.ModbusRequestHandler;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.GlobalEventExecutor;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.logging.Level;
import java.util.logging.Logger;

/* loaded from: com_logistics_fisher_modbus.jar:com/logistics/fisher/modbus/ModbusServer.class */
public class ModbusServer {
    public static final String PROP_CONNECTIONSTATE = "connectionState";
    private final int port;
    private ServerBootstrap bootstrap;
    private Channel parentChannel;
    private CONNECTION_STATES connectionState = CONNECTION_STATES.down;
    private final transient PropertyChangeSupport propertyChangeSupport = new PropertyChangeSupport(this);
    private final ChannelGroup clientChannels = new DefaultChannelGroup(GlobalEventExecutor.INSTANCE);

    /* loaded from: com_logistics_fisher_modbus.jar:com/logistics/fisher/modbus/ModbusServer$CONNECTION_STATES.class */
    public enum CONNECTION_STATES {
        listening,
        down,
        clientsConnected
    }

    public ModbusServer(int port) {
        this.port = port;
    }

    public void setup(ModbusRequestHandler handler) throws ConnectionException {
        handler.setServer(this);
        try {
            final NioEventLoopGroup nioEventLoopGroup = new NioEventLoopGroup();
            final NioEventLoopGroup nioEventLoopGroup2 = new NioEventLoopGroup();
            this.bootstrap = new ServerBootstrap();
            this.bootstrap.group(nioEventLoopGroup, nioEventLoopGroup2).channel(NioServerSocketChannel.class).childHandler(new ModbusChannelInitializer(handler)).option(ChannelOption.SO_BACKLOG, Integer.valueOf(ModbusConstants.ERROR_OFFSET)).childOption(ChannelOption.SO_KEEPALIVE, true);
            this.parentChannel = this.bootstrap.bind(this.port).sync().channel();
            setConnectionState(CONNECTION_STATES.listening);
            this.parentChannel.closeFuture().addListener(new GenericFutureListener<ChannelFuture>() { // from class: com.logistics.fisher.modbus.ModbusServer.1
                public void operationComplete(ChannelFuture future) throws Exception {
                    nioEventLoopGroup2.shutdownGracefully();
                    nioEventLoopGroup.shutdownGracefully();
                    ModbusServer.this.setConnectionState(CONNECTION_STATES.down);
                }
            });
        } catch (Exception ex) {
            setConnectionState(CONNECTION_STATES.down);
            Logger.getLogger(ModbusServer.class.getName()).log(Level.SEVERE, ex.getLocalizedMessage());
            throw new ConnectionException(ex.getLocalizedMessage());
        }
    }

    public CONNECTION_STATES getConnectionState() {
        return this.connectionState;
    }

    public void setConnectionState(CONNECTION_STATES connectionState) {
        CONNECTION_STATES oldConnectionState = this.connectionState;
        this.connectionState = connectionState;
        this.propertyChangeSupport.firePropertyChange("connectionState", oldConnectionState, connectionState);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        this.propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        this.propertyChangeSupport.removePropertyChangeListener(listener);
    }

    public void close() {
        if (this.parentChannel != null) {
            this.parentChannel.close().awaitUninterruptibly();
        }
        this.clientChannels.close().awaitUninterruptibly();
    }

    public ChannelGroup getClientChannels() {
        return this.clientChannels;
    }

    public void addClient(Channel channel) {
        this.clientChannels.add(channel);
        setConnectionState(CONNECTION_STATES.clientsConnected);
    }

    public void removeClient(Channel channel) {
        this.clientChannels.remove(channel);
        setConnectionState(CONNECTION_STATES.clientsConnected);
    }
}
