package com.logistics.fisher.modbus;

import com.logistics.fisher.modbus.entity.exception.ConnectionException;
import com.logistics.fisher.modbus.handler.ModbusChannelInitializer;
import com.logistics.fisher.modbus.handler.ModbusRequestHandler;
import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.GlobalEventExecutor;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ModbusServer {

   public static final String PROP_CONNECTIONSTATE = "connectionState";
   private final int port;
   private ServerBootstrap bootstrap;
   private ModbusServer.CONNECTION_STATES connectionState;
   private final transient PropertyChangeSupport propertyChangeSupport;
   private Channel parentChannel;
   private final ChannelGroup clientChannels;


   public ModbusServer(int port) {
      this.connectionState = ModbusServer.CONNECTION_STATES.down;
      this.propertyChangeSupport = new PropertyChangeSupport(this);
      this.clientChannels = new DefaultChannelGroup(GlobalEventExecutor.INSTANCE);
      this.port = port;
   }

   public void setup(ModbusRequestHandler handler) throws ConnectionException {
      handler.setServer(this);

      try {
         final NioEventLoopGroup ex = new NioEventLoopGroup();
         final NioEventLoopGroup workerGroup = new NioEventLoopGroup();
         this.bootstrap = new ServerBootstrap();
         ((ServerBootstrap)((ServerBootstrap)this.bootstrap.group(ex, workerGroup).channel(NioServerSocketChannel.class)).childHandler(new ModbusChannelInitializer(handler)).option(ChannelOption.SO_BACKLOG, Integer.valueOf(128))).childOption(ChannelOption.SO_KEEPALIVE, Boolean.valueOf(true));
         this.parentChannel = this.bootstrap.bind(this.port).sync().channel();
         this.setConnectionState(ModbusServer.CONNECTION_STATES.listening);
         this.parentChannel.closeFuture().addListener(new GenericFutureListener() {
            public void operationComplete(ChannelFuture future) throws Exception {
               workerGroup.shutdownGracefully();
               ex.shutdownGracefully();
               ModbusServer.this.setConnectionState(ModbusServer.CONNECTION_STATES.down);
            }
         });
      } catch (Exception var4) {
         this.setConnectionState(ModbusServer.CONNECTION_STATES.down);
         Logger.getLogger(ModbusServer.class.getName()).log(Level.SEVERE, var4.getLocalizedMessage());
         throw new ConnectionException(var4.getLocalizedMessage());
      }
   }

   public ModbusServer.CONNECTION_STATES getConnectionState() {
      return this.connectionState;
   }

   public void setConnectionState(ModbusServer.CONNECTION_STATES connectionState) {
      ModbusServer.CONNECTION_STATES oldConnectionState = this.connectionState;
      this.connectionState = connectionState;
      this.propertyChangeSupport.firePropertyChange("connectionState", oldConnectionState, connectionState);
   }

   public void addPropertyChangeListener(PropertyChangeListener listener) {
      this.propertyChangeSupport.addPropertyChangeListener(listener);
   }

   public void removePropertyChangeListener(PropertyChangeListener listener) {
      this.propertyChangeSupport.removePropertyChangeListener(listener);
   }

   public void close() {
      if(this.parentChannel != null) {
         this.parentChannel.close().awaitUninterruptibly();
      }

      this.clientChannels.close().awaitUninterruptibly();
   }

   public ChannelGroup getClientChannels() {
      return this.clientChannels;
   }

   public void addClient(Channel channel) {
      this.clientChannels.add(channel);
      this.setConnectionState(ModbusServer.CONNECTION_STATES.clientsConnected);
   }

   public void removeClient(Channel channel) {
      this.clientChannels.remove(channel);
      this.setConnectionState(ModbusServer.CONNECTION_STATES.clientsConnected);
   }

   public static enum CONNECTION_STATES {

      listening,
      down,
      clientsConnected;
      // $FF: synthetic field
      private static final ModbusServer.CONNECTION_STATES[] $VALUES = new ModbusServer.CONNECTION_STATES[]{listening, down, clientsConnected};


   }
}
