package com.logistics.fisher.modbus;

import com.logistics.fisher.modbus.entity.ModbusFrame;
import com.logistics.fisher.modbus.entity.ModbusFunction;
import com.logistics.fisher.modbus.entity.ModbusHeader;
import com.logistics.fisher.modbus.entity.exception.ConnectionException;
import com.logistics.fisher.modbus.entity.exception.ErrorResponseException;
import com.logistics.fisher.modbus.entity.exception.NoResponseException;
import com.logistics.fisher.modbus.entity.func.WriteSingleCoil;
import com.logistics.fisher.modbus.entity.func.WriteSingleRegister;
import com.logistics.fisher.modbus.entity.func.request.ReadCoilsRequest;
import com.logistics.fisher.modbus.entity.func.request.ReadDiscreteInputsRequest;
import com.logistics.fisher.modbus.entity.func.request.ReadHoldingRegistersRequest;
import com.logistics.fisher.modbus.entity.func.request.ReadInputRegistersRequest;
import com.logistics.fisher.modbus.entity.func.request.WriteMultipleCoilsRequest;
import com.logistics.fisher.modbus.entity.func.request.WriteMultipleRegistersRequest;
import com.logistics.fisher.modbus.entity.func.response.ReadCoilsResponse;
import com.logistics.fisher.modbus.entity.func.response.ReadDiscreteInputsResponse;
import com.logistics.fisher.modbus.entity.func.response.ReadHoldingRegistersResponse;
import com.logistics.fisher.modbus.entity.func.response.ReadInputRegistersResponse;
import com.logistics.fisher.modbus.entity.func.response.WriteMultipleCoilsResponse;
import com.logistics.fisher.modbus.entity.func.response.WriteMultipleRegistersResponse;
import com.logistics.fisher.modbus.handler.ModbusChannelInitializer;
import com.logistics.fisher.modbus.handler.ModbusResponseHandler;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.BitSet;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ModbusClient {

   private ChannelFuture f;
   public static final String PROP_CONNECTIONSTATE = "connectionState";
   private final transient PropertyChangeSupport propertyChangeSupport;
   private final String host;
   private final int port;
   private int lastTransactionIdentifier;
   private Channel channel;
   private short unitIdentifier;
   private short protocolIdentifier;
   private ModbusClient.CONNECTION_STATES connectionState;


   public ModbusClient(String host, int port) {
      this(host, port, (short)1, (short)0);
   }

   public ModbusClient(String host, int port, short unitIdentifier) {
      this(host, port, unitIdentifier, (short)0);
   }

   public ModbusClient(String host, int port, short unitId, short protocolIdentifier) {
      this.f = null;
      this.propertyChangeSupport = new PropertyChangeSupport(this);
      this.lastTransactionIdentifier = 0;
      this.connectionState = ModbusClient.CONNECTION_STATES.notConnected;
      this.host = host;
      this.port = port;
      this.unitIdentifier = unitId;
      this.protocolIdentifier = protocolIdentifier;
   }

   public void setup() throws ConnectionException {
      this.setup((ModbusResponseHandler)null);
   }

   public void setup(ModbusResponseHandler handler) throws ConnectionException {
      NioEventLoopGroup loop = null;

      try {
         loop = new NioEventLoopGroup();
         Bootstrap ex = new Bootstrap();
         ex.group(loop);
         ex.channel(NioSocketChannel.class);
         ex.option(ChannelOption.SO_KEEPALIVE, Boolean.valueOf(true));
         ex.handler(new ModbusChannelInitializer(handler));
         this.setConnectionState(ModbusClient.CONNECTION_STATES.pending);
         this.f = ex.connect(this.host, this.port).sync();
         this.setConnectionState(ModbusClient.CONNECTION_STATES.connected);
         this.channel = this.f.channel();
         Logger.getLogger(ModbusClient.class.getName()).log(Level.INFO, "server connect successful!!!");
         this.f.channel().closeFuture().sync();
      } catch (Exception var7) {
         this.setConnectionState(ModbusClient.CONNECTION_STATES.notConnected);
         Logger.getLogger(ModbusClient.class.getName()).log(Level.SEVERE, var7.getLocalizedMessage());
      } finally {
         if(loop != null) {
            loop.shutdownGracefully();
         }

         this.reconnect();
      }

   }

   private void reconnect() throws ConnectionException {
      Logger.getLogger(ModbusClient.class.getName()).log(Level.SEVERE, "Clear connection resources!!!");
      if(null != this.f && this.f.channel() != null && this.f.channel().isOpen()) {
         this.f.channel().close();
         this.setConnectionState(ModbusClient.CONNECTION_STATES.notConnected);
      }

      try {
         int r = (int)(Math.random() * 70.0D);
         Thread.sleep((long)(r * 1000 + 10000));
      } catch (InterruptedException var2) {
         ;
      }

      Logger.getLogger(ModbusClient.class.getName()).log(Level.SEVERE, "Ready reconnection!!!");
      this.setup();
   }

   public Channel getChannel() {
      return this.channel;
   }

   public void close() {
      if(this.channel != null) {
         this.channel.close().awaitUninterruptibly();
      }

   }

   private synchronized int calculateTransactionIdentifier() {
      if(this.lastTransactionIdentifier < 100) {
         ++this.lastTransactionIdentifier;
      } else {
         this.lastTransactionIdentifier = 1;
      }

      return this.lastTransactionIdentifier;
   }

   public ModbusClient.CONNECTION_STATES getConnectionState() {
      return this.connectionState;
   }

   public void setConnectionState(ModbusClient.CONNECTION_STATES connectionState) {
      ModbusClient.CONNECTION_STATES oldConnectionState = this.connectionState;
      this.connectionState = connectionState;
      this.propertyChangeSupport.firePropertyChange("connectionState", oldConnectionState, connectionState);
   }

   public void addPropertyChangeListener(PropertyChangeListener listener) {
      this.propertyChangeSupport.addPropertyChangeListener(listener);
   }

   public void removePropertyChangeListener(PropertyChangeListener listener) {
      this.propertyChangeSupport.removePropertyChangeListener(listener);
   }

   public int callModbusFunction(ModbusFunction function) throws ConnectionException {
      if(this.channel == null) {
         throw new ConnectionException("Not connected!");
      } else {
         int transactionId = this.calculateTransactionIdentifier();
         int pduLength = function.calculateLength();
         ModbusHeader header = new ModbusHeader(transactionId, this.protocolIdentifier, pduLength, this.unitIdentifier);
         ModbusFrame frame = new ModbusFrame(header, function);
         this.channel.writeAndFlush(frame);
         return transactionId;
      }
   }

   public ModbusFunction callModbusFunctionSync(ModbusFunction function) throws NoResponseException, ErrorResponseException, ConnectionException {
      int transactionId = this.callModbusFunction(function);
      ModbusResponseHandler handler = (ModbusResponseHandler)this.channel.pipeline().get("responseHandler");
      if(handler == null) {
         throw new ConnectionException("Not connected!");
      } else {
         return handler.getResponse(transactionId).getFunction();
      }
   }

   public int writeSingleCoilAsync(int address, boolean state) throws ConnectionException {
      return this.callModbusFunction(new WriteSingleCoil(address, state));
   }

   public int writeSingleRegisterAsync(int address, int value) throws ConnectionException {
      return this.callModbusFunction(new WriteSingleRegister(address, value));
   }

   public int readCoilsAsync(int startAddress, int quantityOfCoils) throws ConnectionException {
      return this.callModbusFunction(new ReadCoilsRequest(startAddress, quantityOfCoils));
   }

   public int readDiscreteInputsAsync(int startAddress, int quantityOfCoils) throws ConnectionException {
      return this.callModbusFunction(new ReadDiscreteInputsRequest(startAddress, quantityOfCoils));
   }

   public int readInputRegistersAsync(int startAddress, int quantityOfInputRegisters) throws ConnectionException {
      return this.callModbusFunction(new ReadInputRegistersRequest(startAddress, quantityOfInputRegisters));
   }

   public int readHoldingRegistersAsync(int startAddress, int quantityOfInputRegisters) throws ConnectionException {
      return this.callModbusFunction(new ReadHoldingRegistersRequest(startAddress, quantityOfInputRegisters));
   }

   public int writeMultipleCoilsAsync(int address, int quantityOfOutputs, BitSet outputsValue) throws ConnectionException {
      return this.callModbusFunction(new WriteMultipleCoilsRequest(address, quantityOfOutputs, outputsValue));
   }

   public int writeMultipleRegistersAsync(int address, int quantityOfRegisters, int[] registers) throws ConnectionException {
      return this.callModbusFunction(new WriteMultipleRegistersRequest(address, quantityOfRegisters, registers));
   }

   public WriteSingleCoil writeSingleCoil(int address, boolean state) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (WriteSingleCoil)this.callModbusFunctionSync(new WriteSingleCoil(address, state));
   }

   public WriteSingleRegister writeSingleRegister(int address, int value) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (WriteSingleRegister)this.callModbusFunctionSync(new WriteSingleRegister(address, value));
   }

   public ReadCoilsResponse readCoils(int startAddress, int quantityOfCoils) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (ReadCoilsResponse)this.callModbusFunctionSync(new ReadCoilsRequest(startAddress, quantityOfCoils));
   }

   public ReadDiscreteInputsResponse readDiscreteInputs(int startAddress, int quantityOfCoils) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (ReadDiscreteInputsResponse)this.callModbusFunctionSync(new ReadDiscreteInputsRequest(startAddress, quantityOfCoils));
   }

   public ReadInputRegistersResponse readInputRegisters(int startAddress, int quantityOfInputRegisters) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (ReadInputRegistersResponse)this.callModbusFunctionSync(new ReadInputRegistersRequest(startAddress, quantityOfInputRegisters));
   }

   public ReadHoldingRegistersResponse readHoldingRegisters(int startAddress, int quantityOfInputRegisters) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (ReadHoldingRegistersResponse)this.callModbusFunctionSync(new ReadHoldingRegistersRequest(startAddress, quantityOfInputRegisters));
   }

   public WriteMultipleCoilsResponse writeMultipleCoils(int address, int quantityOfOutputs, BitSet outputsValue) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (WriteMultipleCoilsResponse)this.callModbusFunctionSync(new WriteMultipleCoilsRequest(address, quantityOfOutputs, outputsValue));
   }

   public WriteMultipleRegistersResponse writeMultipleRegisters(int address, int quantityOfRegisters, int[] registers) throws NoResponseException, ErrorResponseException, ConnectionException {
      return (WriteMultipleRegistersResponse)this.callModbusFunctionSync(new WriteMultipleRegistersRequest(address, quantityOfRegisters, registers));
   }

   public static enum CONNECTION_STATES {

      connected,
      notConnected,
      pending;
      // $FF: synthetic field
      private static final ModbusClient.CONNECTION_STATES[] $VALUES = new ModbusClient.CONNECTION_STATES[]{connected, notConnected, pending};


   }
}
